<?php
session_start();
require_once 'config/config.php';
require_once 'includes/functions.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'message' => 'Geçersiz istek metodu.'], 405);
}

try {
    $quote_token = trim($_POST['quote_token'] ?? '');
    $payment_method = $_POST['payment_method'] ?? '';
    $invoice_type = $_POST['invoice_type'] ?? 'individual';
    
    if (!$quote_token) {
        jsonResponse(['success' => false, 'message' => 'Geçersiz teklif token.']);
    }
    
    // Teklifi al
    $stmt = $pdo->prepare("
        SELECT q.*, qr.id as request_id, qr.customer_id, qr.vehicle_id,
               c.first_name, c.last_name, c.phone, c.email
        FROM quotes q
        JOIN quote_requests qr ON q.quote_request_id = qr.id
        JOIN customers c ON qr.customer_id = c.id
        WHERE q.quote_token = ? AND q.valid_until > NOW()
    ");
    $stmt->execute([$quote_token]);
    $quote = $stmt->fetch();
    
    if (!$quote) {
        jsonResponse(['success' => false, 'message' => 'Teklif bulunamadı veya süresi dolmuş.']);
    }
    
    // Zaten ödenmiş mi kontrol et (orders üzerinden)
    $stmt = $pdo->prepare("
        SELECT p.id 
        FROM payments p 
        JOIN orders o ON p.order_id = o.id 
        WHERE o.quote_id = ? AND p.status = 'completed'
    ");
    $stmt->execute([$quote['id']]);
    if ($stmt->fetch()) {
        jsonResponse(['success' => false, 'message' => 'Bu teklif için ödeme zaten yapılmış.']);
    }
    
    $pdo->beginTransaction();
    
    // Önce sipariş oluştur
    $order_number = generateOrderNumber();
    
    $stmt = $pdo->prepare("
        INSERT INTO orders (quote_id, order_number, customer_id, vehicle_id, 
                           status, subtotal, tax_amount, total_amount, payment_status) 
        VALUES (?, ?, ?, ?, 'pending', ?, ?, ?, 'pending')
    ");
    $stmt->execute([
        $quote['id'],
        $order_number,
        $quote['customer_id'],
        $quote['vehicle_id'],
        $quote['subtotal'] ?? $quote['total_amount'],
        $quote['tax_amount'] ?? 0,
        $quote['total_amount']
    ]);
    
    $order_id = $pdo->lastInsertId();
    
    // Ödeme kaydı oluştur
    $payment_data = [
        'order_id' => $order_id,
        'payment_method' => $payment_method,
        'amount' => $quote['total_amount'],
        'currency' => 'TRY',
        'status' => 'pending'
    ];
    
    if ($payment_method === 'credit_card') {
        // Kredi kartı bilgilerini al
        $card_number = preg_replace('/\s/', '', $_POST['card_number'] ?? '');
        $card_holder = trim($_POST['card_holder'] ?? '');
        $card_expiry = trim($_POST['card_expiry'] ?? '');
        $card_cvv = trim($_POST['card_cvv'] ?? '');
        $installments = intval($_POST['installments'] ?? 1);
        
        // Basit validasyon
        if (strlen($card_number) < 16 || strlen($card_cvv) < 3 || empty($card_holder)) {
            throw new Exception('Kart bilgileri eksik veya hatalı.');
        }
        
        // Demo amaçlı - gerçek ödeme entegrasyonu burada yapılacak
        // Iyzico, PayTR, Stripe vb. entegrasyonu
        
        // Simüle edilmiş ödeme işlemi
        $payment_success = true; // Gerçek entegrasyonda API'den gelecek
        
        if ($payment_success) {
            $payment_data['status'] = 'completed';
            $payment_data['installments'] = $installments;
            $payment_data['card_last_four'] = substr($card_number, -4);
            $payment_data['card_brand'] = 'VISA'; // API'den gelecek
            $payment_data['provider_transaction_id'] = 'DEMO_' . uniqid();
            $payment_data['processed_at'] = date('Y-m-d H:i:s');
        } else {
            $payment_data['status'] = 'failed';
            $payment_data['failure_reason'] = 'Kart işlemi başarısız';
        }
        
    } elseif ($payment_method === 'bank_transfer') {
        // Havale için beklemede bırak
        $payment_data['status'] = 'pending';
    }
    
    // Ödeme kaydını veritabanına ekle
    $stmt = $pdo->prepare("
        INSERT INTO payments (order_id, payment_method, amount, currency, status, transaction_id, processed_at) 
        VALUES (?, ?, ?, ?, ?, ?, ?)
    ");
    $stmt->execute([
        $payment_data['order_id'],
        $payment_data['payment_method'],
        $payment_data['amount'],
        $payment_data['currency'],
        $payment_data['status'],
        $payment_data['provider_transaction_id'] ?? null,
        $payment_data['processed_at'] ?? null
    ]);
    
    $payment_id = $pdo->lastInsertId();
    
    // Ödeme başarılıysa sipariş durumunu güncelle
    if ($payment_data['status'] === 'completed') {
        // Sipariş durumunu güncelle
        $stmt = $pdo->prepare("
            UPDATE orders 
            SET status = 'confirmed', payment_status = 'paid' 
            WHERE id = ?
        ");
        $stmt->execute([$order_id]);
        
        // Teklif kalemlerini sipariş kalemlerine kopyala
        $stmt = $pdo->prepare("
            INSERT INTO order_items (order_id, product_id, name, quantity, unit_price, total_price)
            SELECT ?, product_id, name, quantity, unit_price, line_total
            FROM quote_items 
            WHERE quote_id = ?
        ");
        $stmt->execute([$order_id, $quote['id']]);
        
        // Başvuru ve teklif durumunu güncelle
        $stmt = $pdo->prepare("UPDATE quote_requests SET status = 'PAID' WHERE id = ?");
        $stmt->execute([$quote['request_id']]);
        
        $stmt = $pdo->prepare("UPDATE quotes SET status = 'ACCEPTED', accepted_at = NOW() WHERE id = ?");
        $stmt->execute([$quote['id']]);
        
        // Bildirim gönder
        $message = "Sayın {$quote['first_name']} {$quote['last_name']}, ödemeniz başarıyla alınmıştır. Sipariş No: {$order_number}";
        sendNotification('sms', $quote['phone'], 'Ödeme Onayı', $message);
        
        // Admin bildirim
        $admin_message = "Yeni sipariş: {$order_number} - {$quote['first_name']} {$quote['last_name']} - " . formatMoney($quote['total_amount']);
        sendNotification('email', ADMIN_EMAIL, 'Yeni Sipariş', $admin_message);
        
        // Audit log
        logAudit('payment_completed', 'payments', $payment_id, null, [
            'quote_id' => $quote['id'],
            'amount' => $payment_data['amount'],
            'order_number' => $order_number
        ]);
        
        $pdo->commit();
        
        jsonResponse([
            'success' => true,
            'message' => 'Ödemeniz başarıyla tamamlandı!',
            'redirect' => 'payment_success.php?token=' . $quote_token,
            'order_number' => $order_number
        ]);
        
    } else {
        $pdo->commit();
        
        if ($payment_method === 'bank_transfer') {
            jsonResponse([
                'success' => true,
                'message' => 'Havale bilgileri kaydedildi. Ödeme sonrası siparişiniz oluşturulacaktır.',
                'redirect' => 'payment_pending.php?token=' . $quote_token
            ]);
        } else {
            jsonResponse([
                'success' => false,
                'message' => $payment_data['failure_reason'] ?? 'Ödeme işlemi başarısız.'
            ]);
        }
    }
    
} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    error_log("Payment process error: " . $e->getMessage());
    jsonResponse(['success' => false, 'message' => $e->getMessage()], 500);
}
?>
