<?php
session_start();
require_once '../config/config.php';
require_once '../includes/functions.php';

$pageTitle = 'Teklif Görüntüle';
$currentPage = 'quotes';

if (!isLoggedIn()) {
    redirect('login.php');
}

$quote_id = intval($_GET['id'] ?? 0);

if (!$quote_id) {
    redirect('quotes.php');
}

try {
    // Teklif bilgilerini al
    $stmt = $pdo->prepare("
        SELECT q.*, qr.id as request_id, qr.service_type, qr.problem_description,
               qr.preferred_date, qr.preferred_time, qr.additional_notes,
               c.first_name, c.last_name, c.phone, c.email,
               v.brand, v.model, v.year, v.plate, v.chassis_number,
               u.first_name as created_by_name, u.last_name as created_by_surname
        FROM quotes q
        JOIN quote_requests qr ON q.quote_request_id = qr.id
        JOIN customers c ON qr.customer_id = c.id
        JOIN vehicles v ON qr.vehicle_id = v.id
        LEFT JOIN users u ON q.created_by = u.id
        WHERE q.id = ?
    ");
    $stmt->execute([$quote_id]);
    $quote = $stmt->fetch();
    
    if (!$quote) {
        redirect('quotes.php');
    }
    
    // Teklif kalemlerini al
    $stmt = $pdo->prepare("
        SELECT qi.*, p.name as product_name, p.category_id,
               cat.name as category_name
        FROM quote_items qi
        LEFT JOIN products p ON qi.product_id = p.id
        LEFT JOIN product_categories cat ON p.category_id = cat.id
        WHERE qi.quote_id = ?
        ORDER BY cat.name, p.name, qi.id
    ");
    $stmt->execute([$quote_id]);
    $quote_items = $stmt->fetchAll();
    
} catch (Exception $e) {
    error_log("View quote error: " . $e->getMessage());
    redirect('quotes.php');
}

// Hizmet türü çevirisi
$service_types = [
    'periyodik_bakim' => 'Periyodik Bakım',
    'motor_onarim' => 'Motor Onarımı',
    'kaporta_boya' => 'Kaporta & Boya',
    'genel_onarim' => 'Genel Onarım'
];

// Durum çevirisi
$status_labels = [
    'DRAFT' => ['text' => 'Taslak', 'class' => 'nice-badge-secondary'],
    'SENT' => ['text' => 'Gönderildi', 'class' => 'nice-badge-primary'],
    'VIEWED' => ['text' => 'Görüntülendi', 'class' => 'nice-badge-info'],
    'ACCEPTED' => ['text' => 'Kabul Edildi', 'class' => 'nice-badge-success'],
    'REJECTED' => ['text' => 'Reddedildi', 'class' => 'nice-badge-danger'],
    'EXPIRED' => ['text' => 'Süresi Doldu', 'class' => 'nice-badge-dark']
];
?>
<?php require_once 'includes/header.php'; ?>

<section class="section">
    <div class="pagetitle">
        <h1>Teklif Görüntüle</h1>
        <nav>
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="dashboard.php">Ana Sayfa</a></li>
                <li class="breadcrumb-item"><a href="quotes.php">Teklif Başvuruları</a></li>
                <li class="breadcrumb-item active">Teklif #<?= $quote_id ?></li>
            </ol>
        </nav>
    </div>

    <div class="row">
        <!-- Teklif Bilgileri -->
        <div class="col-lg-8">
            <div class="nice-card">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <h5 class="card-title mb-0">
                            Teklif #<?= $quote_id ?>
                            <span class="nice-badge <?= $status_labels[$quote['status']]['class'] ?? 'nice-badge-secondary' ?> ms-2">
                                <?= $status_labels[$quote['status']]['text'] ?? $quote['status'] ?>
                            </span>
                        </h5>
                        <div class="btn-group">
                            <a href="quotes.php" class="nice-btn nice-btn-outline-secondary">
                                <i class="fas fa-arrow-left"></i> Geri
                            </a>
                            <?php if ($quote['status'] === 'SENT'): ?>
                            <button class="nice-btn nice-btn-primary" onclick="resendQuote(<?= $quote_id ?>)">
                                <i class="fas fa-paper-plane"></i> Tekrar Gönder
                            </button>
                            <?php endif; ?>
                        </div>
                    </div>

                    <!-- Teklif Kalemleri -->
                    <h6 class="mb-3">Teklif Kalemleri</h6>
                    <?php if (empty($quote_items)): ?>
                        <div class="alert alert-info">
                            <i class="fas fa-info-circle me-2"></i>
                            Bu teklifte henüz kalem bulunmuyor.
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="nice-table">
                                <thead>
                                    <tr>
                                        <th>Ürün/Hizmet</th>
                                        <th>Kategori</th>
                                        <th>Miktar</th>
                                        <th>Birim Fiyat</th>
                                        <th>İndirim</th>
                                        <th>KDV</th>
                                        <th>Toplam</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php 
                                    $subtotal = 0;
                                    $total_discount = 0;
                                    $total_tax = 0;
                                    
                                    foreach ($quote_items as $item): 
                                        $item_subtotal = $item['quantity'] * $item['unit_price'];
                                        $item_discount = $item['discount_amount'] ?? 0;
                                        $item_tax = ($item_subtotal - $item_discount) * ($item['tax_rate'] ?? 0) / 100;
                                        $item_total = $item_subtotal - $item_discount + $item_tax;
                                        
                                        $subtotal += $item_subtotal;
                                        $total_discount += $item_discount;
                                        $total_tax += $item_tax;
                                    ?>
                                    <tr>
                                        <td>
                                            <strong><?= h($item['product_name'] ?: $item['description']) ?></strong>
                                            <?php if ($item['description'] && $item['product_name']): ?>
                                                <br><small class="text-muted"><?= h($item['description']) ?></small>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($item['category_name']): ?>
                                                <span class="nice-badge nice-badge-light"><?= h($item['category_name']) ?></span>
                                            <?php else: ?>
                                                <span class="text-muted">-</span>
                                            <?php endif; ?>
                                        </td>
                                        <td><?= number_format($item['quantity'], 2) ?></td>
                                        <td><?= formatMoney($item['unit_price']) ?></td>
                                        <td>
                                            <?php if ($item_discount > 0): ?>
                                                <span class="text-success">-<?= formatMoney($item_discount) ?></span>
                                            <?php else: ?>
                                                <span class="text-muted">-</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if (($item['tax_rate'] ?? 0) > 0): ?>
                                                %<?= number_format($item['tax_rate'], 0) ?>
                                                <br><small class="text-muted"><?= formatMoney($item_tax) ?></small>
                                            <?php else: ?>
                                                <span class="text-muted">-</span>
                                            <?php endif; ?>
                                        </td>
                                        <td><strong><?= formatMoney($item_total) ?></strong></td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                                <tfoot>
                                    <tr class="table-light">
                                        <th colspan="6" class="text-end">Ara Toplam:</th>
                                        <th><?= formatMoney($subtotal) ?></th>
                                    </tr>
                                    <?php if ($total_discount > 0): ?>
                                    <tr class="table-light">
                                        <th colspan="6" class="text-end">Toplam İndirim:</th>
                                        <th class="text-success">-<?= formatMoney($total_discount) ?></th>
                                    </tr>
                                    <?php endif; ?>
                                    <?php if ($total_tax > 0): ?>
                                    <tr class="table-light">
                                        <th colspan="6" class="text-end">Toplam KDV:</th>
                                        <th><?= formatMoney($total_tax) ?></th>
                                    </tr>
                                    <?php endif; ?>
                                    <tr class="table-primary">
                                        <th colspan="6" class="text-end">GENEL TOPLAM:</th>
                                        <th class="fs-5"><?= formatMoney($quote['total_amount']) ?></th>
                                    </tr>
                                </tfoot>
                            </table>
                        </div>
                    <?php endif; ?>

                    <!-- Notlar -->
                    <?php if ($quote['notes']): ?>
                    <div class="mt-4">
                        <h6>Teklif Notları</h6>
                        <div class="alert alert-light">
                            <?= nl2br(h($quote['notes'])) ?>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Yan Panel -->
        <div class="col-lg-4">
            <!-- Müşteri Bilgileri -->
            <div class="nice-card mb-4">
                <div class="card-body">
                    <h6 class="card-title">Müşteri Bilgileri</h6>
                    <div class="customer-info">
                        <div class="info-item">
                            <i class="fas fa-user text-primary me-2"></i>
                            <strong><?= h($quote['first_name'] . ' ' . $quote['last_name']) ?></strong>
                        </div>
                        <div class="info-item">
                            <i class="fas fa-phone text-primary me-2"></i>
                            <a href="tel:<?= h($quote['phone']) ?>"><?= h($quote['phone']) ?></a>
                        </div>
                        <?php if ($quote['email']): ?>
                        <div class="info-item">
                            <i class="fas fa-envelope text-primary me-2"></i>
                            <a href="mailto:<?= h($quote['email']) ?>"><?= h($quote['email']) ?></a>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- Araç Bilgileri -->
            <div class="nice-card mb-4">
                <div class="card-body">
                    <h6 class="card-title">Araç Bilgileri</h6>
                    <div class="vehicle-info">
                        <div class="info-item">
                            <i class="fas fa-car text-primary me-2"></i>
                            <strong><?= h($quote['brand'] . ' ' . $quote['model']) ?></strong>
                        </div>
                        <div class="info-item">
                            <i class="fas fa-calendar text-primary me-2"></i>
                            <?= h($quote['year']) ?> Model
                        </div>
                        <?php if ($quote['plate']): ?>
                        <div class="info-item">
                            <i class="fas fa-id-card text-primary me-2"></i>
                            <?= h($quote['plate']) ?>
                        </div>
                        <?php endif; ?>
                        <?php if ($quote['chassis_number']): ?>
                        <div class="info-item">
                            <i class="fas fa-barcode text-primary me-2"></i>
                            <small><?= h($quote['chassis_number']) ?></small>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- Teklif Detayları -->
            <div class="nice-card mb-4">
                <div class="card-body">
                    <h6 class="card-title">Teklif Detayları</h6>
                    <div class="quote-details">
                        <div class="info-item">
                            <i class="fas fa-calendar-plus text-primary me-2"></i>
                            <strong>Oluşturulma:</strong><br>
                            <small><?= formatDate($quote['created_at'], 'd.m.Y H:i') ?></small>
                        </div>
                        <div class="info-item">
                            <i class="fas fa-calendar-times text-primary me-2"></i>
                            <strong>Geçerlilik:</strong><br>
                            <small><?= formatDate($quote['valid_until'], 'd.m.Y H:i') ?></small>
                        </div>
                        <?php if ($quote['created_by_name']): ?>
                        <div class="info-item">
                            <i class="fas fa-user-tie text-primary me-2"></i>
                            <strong>Hazırlayan:</strong><br>
                            <small><?= h($quote['created_by_name'] . ' ' . $quote['created_by_surname']) ?></small>
                        </div>
                        <?php endif; ?>
                        <?php if ($quote['quote_link']): ?>
                        <div class="info-item">
                            <i class="fas fa-link text-primary me-2"></i>
                            <strong>Müşteri Linki:</strong><br>
                            <a href="<?= h($quote['quote_link']) ?>" target="_blank" class="btn btn-sm btn-outline-primary">
                                <i class="fas fa-external-link-alt"></i> Görüntüle
                            </a>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- Hizmet Bilgileri -->
            <div class="nice-card">
                <div class="card-body">
                    <h6 class="card-title">Hizmet Bilgileri</h6>
                    <div class="service-info">
                        <div class="info-item">
                            <i class="fas fa-tools text-primary me-2"></i>
                            <strong>Hizmet Türü:</strong><br>
                            <span class="nice-badge nice-badge-info">
                                <?= h($service_types[$quote['service_type']] ?? $quote['service_type']) ?>
                            </span>
                        </div>
                        <?php if ($quote['problem_description']): ?>
                        <div class="info-item">
                            <i class="fas fa-comment text-primary me-2"></i>
                            <strong>Problem Açıklaması:</strong><br>
                            <small><?= nl2br(h($quote['problem_description'])) ?></small>
                        </div>
                        <?php endif; ?>
                        <?php if ($quote['preferred_date']): ?>
                        <div class="info-item">
                            <i class="fas fa-calendar-check text-primary me-2"></i>
                            <strong>Tercih Edilen Tarih:</strong><br>
                            <small><?= formatDate($quote['preferred_date'], 'd.m.Y') ?></small>
                            <?php if ($quote['preferred_time']): ?>
                                - <?= h($quote['preferred_time']) ?>
                            <?php endif; ?>
                        </div>
                        <?php endif; ?>
                        <?php if ($quote['additional_notes']): ?>
                        <div class="info-item">
                            <i class="fas fa-sticky-note text-primary me-2"></i>
                            <strong>Ek Notlar:</strong><br>
                            <small><?= nl2br(h($quote['additional_notes'])) ?></small>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>

<style>
.info-item {
    margin-bottom: 1rem;
    padding-bottom: 0.5rem;
    border-bottom: 1px solid #e9ecef;
}

.info-item:last-child {
    margin-bottom: 0;
    border-bottom: none;
}

.customer-info, .vehicle-info, .quote-details, .service-info {
    font-size: 0.9rem;
}
</style>

<script>
// Teklifi tekrar gönderme
function resendQuote(quoteId) {
    if (!confirm('Teklifi tekrar göndermek istediğinizden emin misiniz?')) {
        return;
    }

    fetch('resend_quote.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            quote_id: quoteId
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert('Teklif başarıyla gönderildi!');
        } else {
            alert('Hata: ' + (data.message || 'Bilinmeyen hata'));
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Bir hata oluştu. Lütfen tekrar deneyin.');
    });
}
</script>

<?php require_once 'includes/footer.php'; ?>