<?php
session_start();
require_once '../config/config.php';
require_once '../includes/functions.php';

$pageTitle = 'Raporlar';
$currentPage = 'reports';

if (!isLoggedIn()) {
    redirect('login.php');
}

// Tarih aralığı
$date_from = $_GET['date_from'] ?? date('Y-m-01'); // Bu ayın başı
$date_to = $_GET['date_to'] ?? date('Y-m-t'); // Bu ayın sonu

try {
    // Genel İstatistikler
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_quotes,
            SUM(CASE WHEN status = 'NEW' THEN 1 ELSE 0 END) as new_quotes,
            SUM(CASE WHEN status = 'QUOTE_SENT' THEN 1 ELSE 0 END) as sent_quotes,
            SUM(CASE WHEN status = 'ACCEPTED' THEN 1 ELSE 0 END) as accepted_quotes,
            SUM(CASE WHEN status = 'REJECTED' THEN 1 ELSE 0 END) as rejected_quotes
        FROM quote_requests 
        WHERE DATE(created_at) BETWEEN ? AND ?
    ");
    $stmt->execute([$date_from, $date_to]);
    $quote_stats = $stmt->fetch();

    // Satış İstatistikleri
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_orders,
            SUM(total_amount) as total_revenue,
            AVG(total_amount) as avg_order_value,
            SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed_orders,
            SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending_orders,
            SUM(CASE WHEN status = 'cancelled' THEN 1 ELSE 0 END) as cancelled_orders
        FROM orders 
        WHERE DATE(created_at) BETWEEN ? AND ?
    ");
    $stmt->execute([$date_from, $date_to]);
    $order_stats = $stmt->fetch();

    // Müşteri İstatistikleri
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_customers,
            COUNT(CASE WHEN DATE(created_at) BETWEEN ? AND ? THEN 1 END) as new_customers
        FROM customers
    ");
    $stmt->execute([$date_from, $date_to]);
    $customer_stats = $stmt->fetch();

    // Günlük Teklif Grafiği
    $stmt = $pdo->prepare("
        SELECT 
            DATE(created_at) as date,
            COUNT(*) as count
        FROM quote_requests 
        WHERE DATE(created_at) BETWEEN ? AND ?
        GROUP BY DATE(created_at)
        ORDER BY DATE(created_at)
    ");
    $stmt->execute([$date_from, $date_to]);
    $daily_quotes = $stmt->fetchAll();

    // Günlük Satış Grafiği
    $stmt = $pdo->prepare("
        SELECT 
            DATE(created_at) as date,
            COUNT(*) as count,
            SUM(total_amount) as revenue
        FROM orders 
        WHERE DATE(created_at) BETWEEN ? AND ? AND status = 'completed'
        GROUP BY DATE(created_at)
        ORDER BY DATE(created_at)
    ");
    $stmt->execute([$date_from, $date_to]);
    $daily_sales = $stmt->fetchAll();

    // En Çok Satılan Ürünler
    $stmt = $pdo->prepare("
        SELECT 
            p.name,
            SUM(oi.quantity) as total_quantity,
            SUM(oi.total_amount) as total_revenue
        FROM order_items oi
        JOIN products p ON oi.product_id = p.id
        JOIN orders o ON oi.order_id = o.id
        WHERE DATE(o.created_at) BETWEEN ? AND ? AND o.status = 'completed'
        GROUP BY p.id, p.name
        ORDER BY total_quantity DESC
        LIMIT 10
    ");
    $stmt->execute([$date_from, $date_to]);
    $top_products = $stmt->fetchAll();

    // Müşteri Dağılımı (Şehir bazında)
    $stmt = $pdo->prepare("
        SELECT 
            SUBSTRING_INDEX(TRIM(TRAILING ',' FROM SUBSTRING_INDEX(contact_address, ',', -1)), ' ', -1) as city,
            COUNT(*) as customer_count
        FROM customers 
        WHERE contact_address IS NOT NULL AND contact_address != ''
        GROUP BY city
        ORDER BY customer_count DESC
        LIMIT 10
    ");
    $stmt->execute();
    $customer_cities = $stmt->fetchAll();

} catch (Exception $e) {
    error_log("Reports error: " . $e->getMessage());
    $quote_stats = $order_stats = $customer_stats = [];
    $daily_quotes = $daily_sales = $top_products = $customer_cities = [];
}
?>
<?php require_once 'includes/header.php'; ?>

<section class="section">
    <!-- Filtreler -->
    <div class="nice-card mb-4">
        <div class="card-body">
            <h5 class="card-title">Rapor Filtreleri</h5>
            <form method="GET" class="row g-3">
                <div class="col-md-4">
                    <label for="date_from" class="form-label">Başlangıç Tarihi</label>
                    <input type="date" class="nice-form-control" id="date_from" name="date_from" value="<?= h($date_from) ?>">
                </div>
                <div class="col-md-4">
                    <label for="date_to" class="form-label">Bitiş Tarihi</label>
                    <input type="date" class="nice-form-control" id="date_to" name="date_to" value="<?= h($date_to) ?>">
                </div>
                <div class="col-md-4">
                    <label class="form-label">&nbsp;</label>
                    <div class="d-flex gap-2">
                        <button type="submit" class="nice-btn nice-btn-primary">
                            <i class="fas fa-filter"></i> Filtrele
                        </button>
                        <a href="reports.php" class="nice-btn nice-btn-outline-primary">
                            <i class="fas fa-refresh"></i> Sıfırla
                        </a>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- İstatistik Kartları -->
    <div class="row mb-4">
        <div class="col-xl-3 col-md-6">
            <div class="nice-card info-card sales-card">
                <div class="card-body">
                    <h5 class="card-title">Toplam Teklif</h5>
                    <div class="d-flex align-items-center">
                        <div class="card-icon rounded-circle d-flex align-items-center justify-content-center">
                            <i class="fas fa-clipboard-list"></i>
                        </div>
                        <div class="ps-3">
                            <h6><?= $quote_stats['total_quotes'] ?? 0 ?></h6>
                            <span class="text-success small pt-1 fw-bold">
                                <?= $quote_stats['accepted_quotes'] ?? 0 ?> kabul
                            </span>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6">
            <div class="nice-card info-card revenue-card">
                <div class="card-body">
                    <h5 class="card-title">Toplam Satış</h5>
                    <div class="d-flex align-items-center">
                        <div class="card-icon rounded-circle d-flex align-items-center justify-content-center">
                            <i class="fas fa-lira-sign"></i>
                        </div>
                        <div class="ps-3">
                            <h6><?= formatMoney($order_stats['total_revenue'] ?? 0) ?></h6>
                            <span class="text-info small pt-1 fw-bold">
                                <?= $order_stats['total_orders'] ?? 0 ?> sipariş
                            </span>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6">
            <div class="nice-card info-card customers-card">
                <div class="card-body">
                    <h5 class="card-title">Yeni Müşteri</h5>
                    <div class="d-flex align-items-center">
                        <div class="card-icon rounded-circle d-flex align-items-center justify-content-center">
                            <i class="fas fa-users"></i>
                        </div>
                        <div class="ps-3">
                            <h6><?= $customer_stats['new_customers'] ?? 0 ?></h6>
                            <span class="text-warning small pt-1 fw-bold">
                                Toplam: <?= $customer_stats['total_customers'] ?? 0 ?>
                            </span>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6">
            <div class="nice-card info-card">
                <div class="card-body">
                    <h5 class="card-title">Ortalama Sipariş</h5>
                    <div class="d-flex align-items-center">
                        <div class="card-icon rounded-circle d-flex align-items-center justify-content-center" 
                             style="color: var(--nice-success); background: rgba(25, 135, 84, 0.1);">
                            <i class="fas fa-chart-line"></i>
                        </div>
                        <div class="ps-3">
                            <h6><?= formatMoney($order_stats['avg_order_value'] ?? 0) ?></h6>
                            <span class="text-success small pt-1 fw-bold">
                                Ortalama değer
                            </span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="row">
        <!-- Sol Kolon -->
        <div class="col-lg-8">
            <!-- Günlük Teklif Grafiği -->
            <div class="nice-card mb-4">
                <div class="card-body">
                    <h5 class="card-title">Günlük Teklif Trendi</h5>
                    <div id="quotesChart"></div>
                </div>
            </div>

            <!-- Günlük Satış Grafiği -->
            <div class="nice-card mb-4">
                <div class="card-body">
                    <h5 class="card-title">Günlük Satış Trendi</h5>
                    <div id="salesChart"></div>
                </div>
            </div>

            <!-- En Çok Satılan Ürünler -->
            <div class="nice-card">
                <div class="card-body">
                    <h5 class="card-title">En Çok Satılan Ürünler</h5>
                    <?php if (empty($top_products)): ?>
                        <div class="text-center py-4">
                            <i class="fas fa-box-open fa-3x text-muted mb-3"></i>
                            <p class="text-muted">Bu dönemde satış verisi bulunmuyor.</p>
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="nice-table">
                                <thead>
                                    <tr>
                                        <th>Ürün</th>
                                        <th>Satış Adedi</th>
                                        <th>Toplam Gelir</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($top_products as $product): ?>
                                    <tr>
                                        <td><?= h($product['name']) ?></td>
                                        <td>
                                            <span class="nice-badge nice-badge-primary"><?= $product['total_quantity'] ?></span>
                                        </td>
                                        <td><strong><?= formatMoney($product['total_revenue']) ?></strong></td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Sağ Kolon -->
        <div class="col-lg-4">
            <!-- Teklif Durumu Dağılımı -->
            <div class="nice-card mb-4">
                <div class="card-body">
                    <h5 class="card-title">Teklif Durumu Dağılımı</h5>
                    <div id="quoteStatusChart"></div>
                </div>
            </div>

            <!-- Sipariş Durumu Dağılımı -->
            <div class="nice-card mb-4">
                <div class="card-body">
                    <h5 class="card-title">Sipariş Durumu Dağılımı</h5>
                    <div id="orderStatusChart"></div>
                </div>
            </div>

            <!-- Müşteri Şehir Dağılımı -->
            <div class="nice-card">
                <div class="card-body">
                    <h5 class="card-title">Müşteri Şehir Dağılımı</h5>
                    <?php if (empty($customer_cities)): ?>
                        <div class="text-center py-4">
                            <i class="fas fa-map-marker-alt fa-2x text-muted mb-3"></i>
                            <p class="text-muted">Şehir verisi bulunmuyor.</p>
                        </div>
                    <?php else: ?>
                        <div class="list-group list-group-flush">
                            <?php foreach ($customer_cities as $city): ?>
                            <div class="list-group-item d-flex justify-content-between align-items-center">
                                <span><?= h($city['city']) ?></span>
                                <span class="nice-badge nice-badge-secondary"><?= $city['customer_count'] ?></span>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</section>

<script src="https://cdn.jsdelivr.net/npm/apexcharts"></script>
<script>
document.addEventListener("DOMContentLoaded", () => {
    // Günlük Teklif Grafiği
    const quotesData = <?= json_encode($daily_quotes) ?>;
    const quotesChart = new ApexCharts(document.querySelector("#quotesChart"), {
        series: [{
            name: 'Teklif Sayısı',
            data: quotesData.map(item => ({
                x: item.date,
                y: parseInt(item.count)
            }))
        }],
        chart: {
            type: 'area',
            height: 350,
            toolbar: { show: false }
        },
        colors: ['#4154f1'],
        fill: {
            type: "gradient",
            gradient: {
                shadeIntensity: 1,
                opacityFrom: 0.3,
                opacityTo: 0.4,
                stops: [0, 90, 100]
            }
        },
        dataLabels: { enabled: false },
        stroke: { curve: 'smooth', width: 2 },
        xaxis: { type: 'datetime' },
        tooltip: {
            x: { format: 'dd/MM/yyyy' }
        }
    });
    quotesChart.render();

    // Günlük Satış Grafiği
    const salesData = <?= json_encode($daily_sales) ?>;
    const salesChart = new ApexCharts(document.querySelector("#salesChart"), {
        series: [{
            name: 'Sipariş Sayısı',
            data: salesData.map(item => ({
                x: item.date,
                y: parseInt(item.count)
            }))
        }, {
            name: 'Gelir (₺)',
            data: salesData.map(item => ({
                x: item.date,
                y: parseFloat(item.revenue || 0)
            }))
        }],
        chart: {
            type: 'line',
            height: 350,
            toolbar: { show: false }
        },
        colors: ['#2eca6a', '#ff771d'],
        stroke: { curve: 'smooth', width: 2 },
        xaxis: { type: 'datetime' },
        yaxis: [{
            title: { text: 'Sipariş Sayısı' }
        }, {
            opposite: true,
            title: { text: 'Gelir (₺)' }
        }],
        tooltip: {
            x: { format: 'dd/MM/yyyy' }
        }
    });
    salesChart.render();

    // Teklif Durumu Pasta Grafiği
    const quoteStatusChart = new ApexCharts(document.querySelector("#quoteStatusChart"), {
        series: [
            <?= $quote_stats['new_quotes'] ?? 0 ?>,
            <?= $quote_stats['sent_quotes'] ?? 0 ?>,
            <?= $quote_stats['accepted_quotes'] ?? 0 ?>,
            <?= $quote_stats['rejected_quotes'] ?? 0 ?>
        ],
        chart: {
            type: 'donut',
            height: 300
        },
        labels: ['Yeni', 'Gönderildi', 'Kabul', 'Red'],
        colors: ['#ffc107', '#0dcaf0', '#198754', '#dc3545'],
        legend: {
            position: 'bottom'
        }
    });
    quoteStatusChart.render();

    // Sipariş Durumu Pasta Grafiği
    const orderStatusChart = new ApexCharts(document.querySelector("#orderStatusChart"), {
        series: [
            <?= $order_stats['completed_orders'] ?? 0 ?>,
            <?= $order_stats['pending_orders'] ?? 0 ?>,
            <?= $order_stats['cancelled_orders'] ?? 0 ?>
        ],
        chart: {
            type: 'donut',
            height: 300
        },
        labels: ['Tamamlandı', 'Bekliyor', 'İptal'],
        colors: ['#198754', '#ffc107', '#dc3545'],
        legend: {
            position: 'bottom'
        }
    });
    orderStatusChart.render();
});
</script>

<?php require_once 'includes/footer.php'; ?>
