<?php
session_start();
require_once '../config/config.php';
require_once '../includes/functions.php';

$pageTitle = 'Teklif Başvuruları';
$currentPage = 'quotes';

if (!isLoggedIn()) {
    redirect('login.php');
}

// Sayfalama
$page = intval($_GET['page'] ?? 1);
$limit = 20;
$offset = ($page - 1) * $limit;

// Filtreleme
$status_filter = $_GET['status'] ?? '';
$search = trim($_GET['search'] ?? '');
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';

try {
    // Basit filtreleme
    $where_conditions = [];
    $params = [];
    
    if ($status_filter) {
        $where_conditions[] = "qr.status = ?";
        $params[] = $status_filter;
    }
    
    if ($search) {
        $where_conditions[] = "(c.first_name LIKE ? OR c.last_name LIKE ? OR c.phone LIKE ?)";
        $search_param = "%$search%";
        $params[] = $search_param;
        $params[] = $search_param;
        $params[] = $search_param;
    }
    
    $where_clause = '';
    if (!empty($where_conditions)) {
        $where_clause = 'WHERE ' . implode(' AND ', $where_conditions);
    }

    // Başvuruları al
    $stmt = $pdo->prepare("
        SELECT qr.*, c.first_name, c.last_name, c.phone, c.email, v.brand, v.model, v.year, v.plate,
               q.id as quote_id, q.status as quote_status, q.quote_token, q.quote_link
        FROM quote_requests qr
        JOIN customers c ON qr.customer_id = c.id
        JOIN vehicles v ON qr.vehicle_id = v.id
        LEFT JOIN quotes q ON qr.id = q.quote_request_id
        $where_clause
        ORDER BY qr.created_at DESC
        LIMIT $limit OFFSET $offset
    ");
    $stmt->execute($params);
    $requests = $stmt->fetchAll();

    // Toplam kayıt sayısı
    $count_stmt = $pdo->prepare("
        SELECT COUNT(*) as total 
        FROM quote_requests qr
        JOIN customers c ON qr.customer_id = c.id
        JOIN vehicles v ON qr.vehicle_id = v.id
        $where_clause
    ");
    $count_stmt->execute($params);
    $total_records = $count_stmt->fetch()['total'];

    // Sayfalama bilgileri
    $total_pages = ceil($total_records / $limit);
    $pagination = [
        'current_page' => $page,
        'total_pages' => $total_pages,
        'total_records' => $total_records,
        'limit' => $limit
    ];

    // Durum istatistikleri
    $stmt = $pdo->query("
        SELECT status, COUNT(*) as count 
        FROM quote_requests 
        GROUP BY status
    ");
    $status_stats = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

} catch (Exception $e) {
    error_log("Quotes page error: " . $e->getMessage());
    // Debug için hatayı ekrana da yazdır
    echo "<!-- Debug: " . $e->getMessage() . " -->";
    $requests = [];
    $pagination = ['total_pages' => 0, 'current_page' => 1, 'total_records' => 0];
    $status_stats = [];
}

// Hizmet türü çevirisi
$service_types = [
    'periyodik_bakim' => 'Periyodik Bakım',
    'motor_onarim' => 'Motor Onarımı',
    'kaporta_boya' => 'Kaporta & Boya',
    'genel_onarim' => 'Genel Onarım'
];

// Durum çevirisi
$status_labels = [
    'NEW' => ['text' => 'Yeni', 'class' => 'nice-badge-warning'],
    'IN_PROGRESS' => ['text' => 'İşlemde', 'class' => 'nice-badge-info'],
    'QUOTE_SENT' => ['text' => 'Teklif Gönderildi', 'class' => 'nice-badge-primary'],
    'ACCEPTED' => ['text' => 'Kabul Edildi', 'class' => 'nice-badge-success'],
    'REJECTED' => ['text' => 'Reddedildi', 'class' => 'nice-badge-danger'],
    'CANCELLED' => ['text' => 'İptal Edildi', 'class' => 'nice-badge-secondary'],
    'EXPIRED' => ['text' => 'Süresi Doldu', 'class' => 'nice-badge-dark']
];
?>
<?php require_once 'includes/header.php'; ?>

<section class="section">
    <div class="pagetitle">
        <h1>Teklif Başvuruları</h1>
        <nav>
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="dashboard.php">Ana Sayfa</a></li>
                <li class="breadcrumb-item active">Teklif Başvuruları</li>
            </ol>
        </nav>
    </div>

    <!-- İstatistik Kartları -->
    <div class="row mb-4">
        <div class="col-xl-3 col-md-6">
            <div class="nice-card info-card sales-card">
                <div class="card-body">
                    <h5 class="card-title">Toplam Başvuru</h5>
                    <div class="d-flex align-items-center">
                        <div class="card-icon rounded-circle d-flex align-items-center justify-content-center">
                            <i class="fas fa-file-alt"></i>
                        </div>
                        <div class="ps-3">
                            <h6><?= array_sum($status_stats) ?></h6>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6">
            <div class="nice-card info-card revenue-card">
                <div class="card-body">
                    <h5 class="card-title">Yeni Başvuru</h5>
                    <div class="d-flex align-items-center">
                        <div class="card-icon rounded-circle d-flex align-items-center justify-content-center">
                            <i class="fas fa-plus-circle"></i>
                        </div>
                        <div class="ps-3">
                            <h6><?= $status_stats['NEW'] ?? 0 ?></h6>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6">
            <div class="nice-card info-card customers-card">
                <div class="card-body">
                    <h5 class="card-title">İşlemde</h5>
                    <div class="d-flex align-items-center">
                        <div class="card-icon rounded-circle d-flex align-items-center justify-content-center">
                            <i class="fas fa-cog"></i>
                        </div>
                        <div class="ps-3">
                            <h6><?= $status_stats['IN_PROGRESS'] ?? 0 ?></h6>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6">
            <div class="nice-card info-card">
                <div class="card-body">
                    <h5 class="card-title">Tamamlanan</h5>
                    <div class="d-flex align-items-center">
                        <div class="card-icon rounded-circle d-flex align-items-center justify-content-center" 
                             style="color: var(--nice-success); background: rgba(25, 135, 84, 0.1);">
                            <i class="fas fa-check-circle"></i>
                        </div>
                        <div class="ps-3">
                            <h6><?= ($status_stats['ACCEPTED'] ?? 0) + ($status_stats['QUOTE_SENT'] ?? 0) ?></h6>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Filtreler -->
    <div class="nice-card mb-4">
        <div class="card-body">
            <h5 class="card-title">Filtreler</h5>
            <form method="GET" class="row g-3">
                <div class="col-md-3">
                    <label for="search" class="form-label">Arama</label>
                    <input type="text" class="nice-form-control" id="search" name="search" 
                           value="<?= h($search) ?>" placeholder="Ad, soyad, telefon, araç...">
                </div>
                
                <div class="col-md-2">
                    <label for="status" class="form-label">Durum</label>
                    <select class="nice-form-control" id="status" name="status">
                        <option value="">Tümü</option>
                        <option value="NEW" <?= $status_filter === 'NEW' ? 'selected' : '' ?>>Yeni Başvuru</option>
                        <option value="IN_PROGRESS" <?= $status_filter === 'IN_PROGRESS' ? 'selected' : '' ?>>Teklif Hazırlanıyor</option>
                        <option value="QUOTE_SENT" <?= $status_filter === 'QUOTE_SENT' ? 'selected' : '' ?>>Teklif Gönderildi</option>
                        <option value="QUOTE_VIEWED" <?= $status_filter === 'QUOTE_VIEWED' ? 'selected' : '' ?>>Teklif Görüntülendi</option>
                        <option value="ACCEPTED" <?= $status_filter === 'ACCEPTED' ? 'selected' : '' ?>>Kabul Edildi</option>
                        <option value="PAID" <?= $status_filter === 'PAID' ? 'selected' : '' ?>>Ödeme Yapıldı</option>
                        <option value="REJECTED" <?= $status_filter === 'REJECTED' ? 'selected' : '' ?>>Reddedildi</option>
                        <option value="CANCELLED" <?= $status_filter === 'CANCELLED' ? 'selected' : '' ?>>İptal Edildi</option>
                    </select>
                </div>
                
                <div class="col-md-2">
                    <label for="date_from" class="form-label">Başlangıç</label>
                    <input type="date" class="nice-form-control" id="date_from" name="date_from" value="<?= h($date_from) ?>">
                </div>
                
                <div class="col-md-2">
                    <label for="date_to" class="form-label">Bitiş</label>
                    <input type="date" class="nice-form-control" id="date_to" name="date_to" value="<?= h($date_to) ?>">
                </div>
                
                <div class="col-md-3">
                    <label class="form-label">&nbsp;</label>
                    <div class="d-flex gap-2">
                        <button type="submit" class="nice-btn nice-btn-primary">
                            <i class="fas fa-search"></i> Filtrele
                        </button>
                        <a href="quotes.php" class="nice-btn nice-btn-outline-secondary">
                            <i class="fas fa-times"></i> Temizle
                        </a>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Başvurular Listesi -->
    <div class="nice-card">
        <div class="card-body">
            <div class="d-flex justify-content-between align-items-center mb-3">
                <h5 class="card-title mb-0">
                    Başvurular 
                    <span class="nice-badge nice-badge-primary"><?= $total_records ?></span>
                </h5>
            </div>

            <?php if (empty($requests)): ?>
                <div class="text-center py-5">
                    <i class="fas fa-inbox fa-3x text-muted mb-3"></i>
                    <h5 class="text-muted">Henüz başvuru bulunmuyor</h5>
                    <p class="text-muted">Yeni başvurular burada görünecektir.</p>
                </div>
            <?php else: ?>
                <div class="table-responsive">
                    <table class="nice-table table-hover">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Müşteri</th>
                                <th>İletişim</th>
                                <th>Araç</th>
                                <th>Hizmet Türü</th>
                                <th>Durum</th>
                                <th>Tarih</th>
                                <th>İşlemler</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($requests as $request): ?>
                            <tr>
                                <td><strong>#<?= $request['id'] ?></strong></td>
                                <td>
                                    <strong><?= h($request['first_name'] . ' ' . $request['last_name']) ?></strong>
                                </td>
                                <td>
                                    <div>
                                        <i class="fas fa-phone text-muted me-1"></i>
                                        <small><?= h($request['phone']) ?></small>
                                    </div>
                                    <?php if ($request['email']): ?>
                                    <div>
                                        <i class="fas fa-envelope text-muted me-1"></i>
                                        <small><?= h($request['email']) ?></small>
                                    </div>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <strong><?= h($request['brand'] . ' ' . $request['model']) ?></strong>
                                    <br>
                                    <small class="text-muted">
                                        <?= h($request['year']) ?>
                                        <?php if ($request['plate']): ?>
                                            • <?= h($request['plate']) ?>
                                        <?php endif; ?>
                                    </small>
                                </td>
                                <td>
                                    <span class="nice-badge nice-badge-info">
                                        <?= h($service_types[$request['service_type']] ?? $request['service_type']) ?>
                                    </span>
                                </td>
                                <td>
                                    <span class="nice-badge <?= $status_labels[$request['status']]['class'] ?? 'nice-badge-secondary' ?>">
                                        <?= $status_labels[$request['status']]['text'] ?? $request['status'] ?>
                                    </span>
                                </td>
                                <td>
                                    <small><?= formatDate($request['created_at'], 'd.m.Y H:i') ?></small>
                                </td>
                                <td>
                                    <div class="btn-group">
                                        <!-- Detay Butonu -->
                                        <a href="quote_detail.php?id=<?= $request['id'] ?>" 
                                           class="nice-btn nice-btn-outline-primary btn-sm" 
                                           title="Detay Görüntüle">
                                            <i class="fas fa-eye"></i>
                                        </a>

                                        <!-- Teklif Hazırla Butonu -->
                                        <?php if ($request['status'] === 'NEW' && !$request['quote_id']): ?>
                                        <a href="create_quote.php?request_id=<?= $request['id'] ?>" 
                                           class="nice-btn nice-btn-success btn-sm" 
                                           title="Teklif Hazırla">
                                            <i class="fas fa-plus"></i>
                                        </a>
                                        <?php endif; ?>

                                        <!-- Teklif Görüntüle Butonu -->
                                        <?php if ($request['quote_id']): ?>
                                        <a href="view_quote.php?id=<?= $request['quote_id'] ?>" 
                                           class="nice-btn nice-btn-info btn-sm" 
                                           title="Teklif Görüntüle">
                                            <i class="fas fa-file-invoice"></i>
                                        </a>
                                        <?php endif; ?>

                                        <!-- Teklif Linki Kopyala Butonu -->
                                        <?php if ($request['quote_token']): ?>
                                        <button type="button" 
                                                class="nice-btn nice-btn-warning btn-sm" 
                                                onclick="copyQuoteLink('<?= $request['quote_token'] ?>')"
                                                title="Teklif Linkini Kopyala">
                                            <i class="fas fa-link"></i>
                                        </button>
                                        <?php endif; ?>

                                        <!-- İşlemler Dropdown -->
                                        <div class="dropdown">
                                            <button class="nice-btn nice-btn-outline-secondary btn-sm dropdown-toggle" 
                                                    type="button" 
                                                    data-bs-toggle="dropdown" 
                                                    aria-expanded="false">
                                                <i class="fas fa-ellipsis-v"></i>
                                            </button>
                                            <ul class="dropdown-menu dropdown-menu-end">
                                                <?php if (in_array($request['status'], ['NEW', 'IN_PROGRESS'])): ?>
                                                <li>
                                                    <button class="dropdown-item" onclick="updateStatus(<?= $request['id'] ?>, 'IN_PROGRESS')">
                                                        <i class="fas fa-play text-warning me-2"></i>İşleme Al
                                                    </button>
                                                </li>
                                                <li><hr class="dropdown-divider"></li>
                                                <?php endif; ?>
                                                
                                                <?php if ($request['quote_id'] && $request['status'] === 'QUOTE_SENT'): ?>
                                                <li>
                                                    <button class="dropdown-item" onclick="resendQuote(<?= $request['quote_id'] ?>)">
                                                        <i class="fas fa-paper-plane text-info me-2"></i>Teklifi Tekrar Gönder
                                                    </button>
                                                </li>
                                                <li><hr class="dropdown-divider"></li>
                                                <?php endif; ?>
                                                
                                                <!-- Teklif Düzenleme -->
                                                <?php if ($request['quote_id'] && in_array($request['status'], ['NEW', 'IN_PROGRESS', 'QUOTE_SENT'])): ?>
                                                <li>
                                                    <a class="dropdown-item" href="edit_quote.php?id=<?= $request['quote_id'] ?>">
                                                        <i class="fas fa-edit text-info me-2"></i>Teklif Düzenle
                                                    </a>
                                                </li>
                                                <?php endif; ?>
                                                
                                                <!-- Teklif Silme -->
                                                <?php if ($request['quote_id'] && in_array($request['status'], ['NEW', 'IN_PROGRESS', 'QUOTE_SENT'])): ?>
                                                <li>
                                                    <button class="dropdown-item text-warning" onclick="deleteQuote(<?= $request['quote_id'] ?>, '<?= h($request['first_name'] . ' ' . $request['last_name']) ?>')">
                                                        <i class="fas fa-trash text-warning me-2"></i>Teklif Sil
                                                    </button>
                                                </li>
                                                <li><hr class="dropdown-divider"></li>
                                                <?php endif; ?>
                                                
                                                <?php if (in_array($request['status'], ['NEW', 'IN_PROGRESS'])): ?>
                                                <li>
                                                    <button class="dropdown-item text-danger" onclick="cancelRequest(<?= $request['id'] ?>)">
                                                        <i class="fas fa-times text-danger me-2"></i>İptal Et
                                                    </button>
                                                </li>
                                                <?php endif; ?>
                                            </ul>
                                        </div>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Sayfalama -->
                <?php if ($pagination['total_pages'] > 1): ?>
                <nav aria-label="Sayfa navigasyonu" class="mt-4">
                    <ul class="pagination justify-content-center">
                        <?php if ($pagination['current_page'] > 1): ?>
                        <li class="page-item">
                            <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $pagination['current_page'] - 1])) ?>">
                                <i class="fas fa-chevron-left"></i>
                            </a>
                        </li>
                        <?php endif; ?>

                        <?php for ($i = max(1, $pagination['current_page'] - 2); $i <= min($pagination['total_pages'], $pagination['current_page'] + 2); $i++): ?>
                        <li class="page-item <?= $i === $pagination['current_page'] ? 'active' : '' ?>">
                            <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $i])) ?>">
                                <?= $i ?>
                            </a>
                        </li>
                        <?php endfor; ?>

                        <?php if ($pagination['current_page'] < $pagination['total_pages']): ?>
                        <li class="page-item">
                            <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $pagination['current_page'] + 1])) ?>">
                                <i class="fas fa-chevron-right"></i>
                            </a>
                        </li>
                        <?php endif; ?>
                    </ul>
                </nav>
                <?php endif; ?>
            <?php endif; ?>
        </div>
    </div>
</section>

<script>
// Durum güncelleme
function updateStatus(requestId, status) {
    if (!confirm('Bu başvurunun durumunu güncellemek istediğinizden emin misiniz?')) {
        return;
    }

    fetch('update_quote_status.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            request_id: requestId,
            status: status
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            location.reload();
        } else {
            alert('Hata: ' + (data.message || 'Bilinmeyen hata'));
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Bir hata oluştu. Lütfen tekrar deneyin.');
    });
}

// Başvuru iptal etme
function cancelRequest(requestId) {
    if (!confirm('Bu başvuruyu iptal etmek istediğinizden emin misiniz?')) {
        return;
    }

    updateStatus(requestId, 'CANCELLED');
}

// Teklifi tekrar gönderme
function resendQuote(quoteId) {
    if (!confirm('Teklifi tekrar göndermek istediğinizden emin misiniz?')) {
        return;
    }

    fetch('resend_quote.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            quote_id: quoteId
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert('Teklif başarıyla gönderildi!');
        } else {
            alert('Hata: ' + (data.message || 'Bilinmeyen hata'));
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Bir hata oluştu. Lütfen tekrar deneyin.');
    });
}

// Teklif linkini kopyala
function copyQuoteLink(token) {
    const baseUrl = window.location.protocol + '//' + window.location.host;
    const quoteUrl = baseUrl + '/view_quote.php?token=' + token;
    
    // Clipboard API kullan
    if (navigator.clipboard && window.isSecureContext) {
        navigator.clipboard.writeText(quoteUrl).then(() => {
            showNotification('Teklif linki panoya kopyalandı!', 'success');
        }).catch(err => {
            console.error('Clipboard error:', err);
            fallbackCopyTextToClipboard(quoteUrl);
        });
    } else {
        // Fallback method
        fallbackCopyTextToClipboard(quoteUrl);
    }
}

// Fallback kopyalama metodu
function fallbackCopyTextToClipboard(text) {
    const textArea = document.createElement("textarea");
    textArea.value = text;
    
    // Görünmez yap
    textArea.style.top = "0";
    textArea.style.left = "0";
    textArea.style.position = "fixed";
    textArea.style.opacity = "0";
    
    document.body.appendChild(textArea);
    textArea.focus();
    textArea.select();
    
    try {
        const successful = document.execCommand('copy');
        if (successful) {
            showNotification('Teklif linki panoya kopyalandı!', 'success');
        } else {
            showQuoteLinkModal(text);
        }
    } catch (err) {
        console.error('Fallback copy error:', err);
        showQuoteLinkModal(text);
    }
    
    document.body.removeChild(textArea);
}

// Link gösterme modalı
function showQuoteLinkModal(url) {
    const modal = document.createElement('div');
    modal.className = 'modal fade';
    modal.innerHTML = `
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Teklif Linki</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <p>Aşağıdaki linki kopyalayın:</p>
                    <div class="input-group">
                        <input type="text" class="form-control" value="${url}" readonly id="quoteLinkInput">
                        <button class="btn btn-outline-secondary" type="button" onclick="selectAndCopy()">
                            <i class="fas fa-copy"></i> Kopyala
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
    const bsModal = new bootstrap.Modal(modal);
    bsModal.show();
    
    // Modal kapandığında temizle
    modal.addEventListener('hidden.bs.modal', () => {
        document.body.removeChild(modal);
    });
}

// Input seç ve kopyala
function selectAndCopy() {
    const input = document.getElementById('quoteLinkInput');
    input.select();
    input.setSelectionRange(0, 99999); // Mobil için
    
    try {
        document.execCommand('copy');
        showNotification('Link kopyalandı!', 'success');
    } catch (err) {
        console.error('Copy error:', err);
    }
}

// Bildirim göster
function showNotification(message, type = 'info') {
    const alertDiv = document.createElement('div');
    alertDiv.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
    alertDiv.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
    alertDiv.innerHTML = `
        <i class="fas fa-${type === 'success' ? 'check-circle' : 'info-circle'} me-2"></i>${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    document.body.appendChild(alertDiv);
    
    // 3 saniye sonra otomatik kaldır
    setTimeout(() => {
        if (alertDiv.parentNode) {
            alertDiv.remove();
        }
    }, 3000);
}

// Teklif silme
function deleteQuote(quoteId, customerName) {
    if (!confirm(`"${customerName}" müşterisinin teklifini silmek istediğinizden emin misiniz?\n\nBu işlem geri alınamaz!`)) {
        return;
    }

    fetch('delete_quote.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            quote_id: quoteId
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification('Teklif başarıyla silindi!', 'success');
            // Sayfayı yenile
            setTimeout(() => {
                location.reload();
            }, 1500);
        } else {
            alert('Hata: ' + (data.message || 'Bilinmeyen hata'));
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Bir hata oluştu. Lütfen tekrar deneyin.');
    });
}

// Otomatik yenileme (5 dakikada bir)
setInterval(() => {
    if (document.visibilityState === 'visible') {
        location.reload();
    }
}, 300000);
</script>

<?php require_once 'includes/footer.php'; ?>