<?php
session_start();
require_once '../config/config.php';
require_once '../includes/functions.php';

$pageTitle = 'Siparişler';
$currentPage = 'orders';

// Sayfalama
$page = intval($_GET['page'] ?? 1);
$limit = 20;
$offset = ($page - 1) * $limit;

// Filtreleme
$status_filter = $_GET['status'] ?? '';
$search = trim($_GET['search'] ?? '');
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';

try {
    // Siparişleri al
    $where_conditions = ['1=1'];
    $params = [];

    if ($status_filter) {
        $where_conditions[] = "o.status = ?";
        $params[] = $status_filter;
    }

    if ($search) {
        $where_conditions[] = "(o.order_number LIKE ? OR c.first_name LIKE ? OR c.last_name LIKE ? OR c.phone LIKE ?)";
        $params[] = "%$search%";
        $params[] = "%$search%";
        $params[] = "%$search%";
        $params[] = "%$search%";
    }

    if ($date_from) {
        $where_conditions[] = "DATE(o.created_at) >= ?";
        $params[] = $date_from;
    }

    if ($date_to) {
        $where_conditions[] = "DATE(o.created_at) <= ?";
        $params[] = $date_to;
    }

    $where_clause = implode(' AND ', $where_conditions);

    // Toplam kayıt sayısı
    $count_sql = "
        SELECT COUNT(*) as total 
        FROM orders o
        JOIN customers c ON o.customer_id = c.id
        WHERE $where_clause
    ";
    $stmt = $pdo->prepare($count_sql);
    $stmt->execute($params);
    $total_records = $stmt->fetch()['total'];

    // Siparişleri getir
    $sql = "
        SELECT o.*, c.first_name, c.last_name, c.phone, c.email,
               v.brand, v.model, v.year,
               q.quote_number,
               p.status as payment_status
        FROM orders o
        JOIN customers c ON o.customer_id = c.id
        JOIN vehicles v ON o.vehicle_id = v.id
        LEFT JOIN quotes q ON o.quote_id = q.id
        LEFT JOIN payments p ON q.id = p.quote_id AND p.status = 'completed'
        WHERE $where_clause
        ORDER BY o.created_at DESC
        LIMIT $limit OFFSET $offset
    ";

    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $orders = $stmt->fetchAll();

    $pagination = calculatePagination($total_records, $page, $limit);

    // Durum istatistikleri
    $stats_sql = "
        SELECT 
            status,
            COUNT(*) as count,
            SUM(total_amount) as total_amount
        FROM orders 
        GROUP BY status
    ";
    $stmt = $pdo->query($stats_sql);
    $status_stats = [];
    while ($row = $stmt->fetch()) {
        $status_stats[$row['status']] = $row;
    }
    
} catch (Exception $e) {
    error_log("Orders page error: " . $e->getMessage());
    $orders = [];
    $pagination = ['total_pages' => 0, 'current_page' => 1];
    $status_stats = [];
}
?>
<?php require_once 'includes/header.php'; ?>
        <section class="section">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2"><i class="fas fa-shopping-cart"></i> Sipariş Yönetimi</h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <div class="btn-group me-2">
                            <button type="button" class="btn btn-sm btn-outline-secondary" onclick="location.reload()">
                                <i class="fas fa-sync-alt"></i> Yenile
                            </button>
                            <button type="button" class="btn btn-sm btn-outline-secondary" onclick="exportOrders()">
                                <i class="fas fa-download"></i> Dışa Aktar
                            </button>
                        </div>
                    </div>
                </div>

                <!-- Durum İstatistikleri -->
                <div class="row mb-4">
                    <div class="col-md-3 mb-2">
                        <div class="stats-card">
                            <div class="text-center">
                                <div class="stats-number text-warning"><?= $status_stats['pending']['count'] ?? 0 ?></div>
                                <div class="stats-label">Bekleyen</div>
                                <small class="text-muted"><?= formatMoney($status_stats['pending']['total_amount'] ?? 0) ?></small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3 mb-2">
                        <div class="stats-card">
                            <div class="text-center">
                                <div class="stats-number text-info"><?= $status_stats['confirmed']['count'] ?? 0 ?></div>
                                <div class="stats-label">Onaylandı</div>
                                <small class="text-muted"><?= formatMoney($status_stats['confirmed']['total_amount'] ?? 0) ?></small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3 mb-2">
                        <div class="stats-card">
                            <div class="text-center">
                                <div class="stats-number text-primary"><?= $status_stats['in_progress']['count'] ?? 0 ?></div>
                                <div class="stats-label">İşlemde</div>
                                <small class="text-muted"><?= formatMoney($status_stats['in_progress']['total_amount'] ?? 0) ?></small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3 mb-2">
                        <div class="stats-card">
                            <div class="text-center">
                                <div class="stats-number text-success"><?= $status_stats['completed']['count'] ?? 0 ?></div>
                                <div class="stats-label">Tamamlandı</div>
                                <small class="text-muted"><?= formatMoney($status_stats['completed']['total_amount'] ?? 0) ?></small>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Filtreler -->
                <div class="card mb-4">
                    <div class="card-body">
                        <form method="GET" class="row g-3">
                            <div class="col-md-3">
                                <label for="search" class="form-label">Arama</label>
                                <input type="text" class="form-control" id="search" name="search" 
                                       value="<?= h($search) ?>" placeholder="Sipariş no, müşteri adı">
                            </div>
                            <div class="col-md-2">
                                <label for="status" class="form-label">Durum</label>
                                <select class="form-select" id="status" name="status">
                                    <option value="">Tüm Durumlar</option>
                                    <option value="pending" <?= $status_filter === 'pending' ? 'selected' : '' ?>>Bekleyen</option>
                                    <option value="confirmed" <?= $status_filter === 'confirmed' ? 'selected' : '' ?>>Onaylandı</option>
                                    <option value="in_progress" <?= $status_filter === 'in_progress' ? 'selected' : '' ?>>İşlemde</option>
                                    <option value="completed" <?= $status_filter === 'completed' ? 'selected' : '' ?>>Tamamlandı</option>
                                    <option value="cancelled" <?= $status_filter === 'cancelled' ? 'selected' : '' ?>>İptal</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label for="date_from" class="form-label">Başlangıç</label>
                                <input type="date" class="form-control" id="date_from" name="date_from" value="<?= h($date_from) ?>">
                            </div>
                            <div class="col-md-2">
                                <label for="date_to" class="form-label">Bitiş</label>
                                <input type="date" class="form-control" id="date_to" name="date_to" value="<?= h($date_to) ?>">
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">&nbsp;</label>
                                <div class="d-grid">
                                    <button type="submit" class="btn btn-primary">
                                        <i class="fas fa-search"></i> Filtrele
                                    </button>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Sipariş Listesi -->
                <div class="card">
                    <div class="card-header">
                        <h5><i class="fas fa-list"></i> Sipariş Listesi (<?= $total_records ?> kayıt)</h5>
                    </div>
                    <div class="card-body">
                        <?php if (empty($orders)): ?>
                            <div class="text-center py-5">
                                <i class="fas fa-shopping-cart fa-3x text-muted mb-3"></i>
                                <p class="text-muted">Henüz sipariş bulunmuyor.</p>
                            </div>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>Sipariş No</th>
                                            <th>Müşteri</th>
                                            <th>Araç</th>
                                            <th>Tutar</th>
                                            <th>Durum</th>
                                            <th>Ödeme</th>
                                            <th>Tarih</th>
                                            <th>İşlemler</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($orders as $order): ?>
                                        <tr>
                                            <td>
                                                <strong><?= h($order['order_number']) ?></strong>
                                                <?php if ($order['quote_number']): ?>
                                                    <br><small class="text-muted">Teklif: <?= h($order['quote_number']) ?></small>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <div>
                                                    <strong><?= h($order['first_name'] . ' ' . $order['last_name']) ?></strong>
                                                    <br><small class="text-muted">
                                                        <i class="fas fa-phone"></i> <?= h($order['phone']) ?>
                                                    </small>
                                                </div>
                                            </td>
                                            <td>
                                                <div>
                                                    <strong><?= h($order['brand'] . ' ' . $order['model']) ?></strong>
                                                    <br><small class="text-muted"><?= h($order['year']) ?></small>
                                                </div>
                                            </td>
                                            <td>
                                                <strong class="text-success"><?= formatMoney($order['total_amount']) ?></strong>
                                            </td>
                                            <td>
                                                <?php
                                                $statusClass = [
                                                    'pending' => 'badge bg-warning',
                                                    'confirmed' => 'badge bg-info',
                                                    'in_progress' => 'badge bg-primary',
                                                    'completed' => 'badge bg-success',
                                                    'cancelled' => 'badge bg-danger'
                                                ];
                                                $statusText = [
                                                    'pending' => 'Bekleyen',
                                                    'confirmed' => 'Onaylandı',
                                                    'in_progress' => 'İşlemde',
                                                    'completed' => 'Tamamlandı',
                                                    'cancelled' => 'İptal'
                                                ];
                                                ?>
                                                <span class="<?= $statusClass[$order['status']] ?? 'badge bg-secondary' ?>">
                                                    <?= $statusText[$order['status']] ?? $order['status'] ?>
                                                </span>
                                            </td>
                                            <td>
                                                <?php if ($order['payment_status'] === 'completed'): ?>
                                                    <span class="badge bg-success">
                                                        <i class="fas fa-check"></i> Ödendi
                                                    </span>
                                                <?php else: ?>
                                                    <span class="badge bg-warning">
                                                        <i class="fas fa-clock"></i> Bekleyen
                                                    </span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <small><?= formatDate($order['created_at'], 'd.m.Y H:i') ?></small>
                                                <?php if ($order['scheduled_date']): ?>
                                                    <br><small class="text-info">
                                                        <i class="fas fa-calendar"></i> <?= formatDate($order['scheduled_date'], 'd.m.Y') ?>
                                                    </small>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <div class="btn-group btn-group-sm">
                                                    <a href="order_detail.php?id=<?= $order['id'] ?>" 
                                                       class="btn btn-outline-primary" title="Detay">
                                                        <i class="fas fa-eye"></i>
                                                    </a>
                                                    <div class="btn-group btn-group-sm">
                                                        <button type="button" class="btn btn-outline-secondary dropdown-toggle" 
                                                                data-bs-toggle="dropdown">
                                                            <i class="fas fa-ellipsis-v"></i>
                                                        </button>
                                                        <ul class="dropdown-menu">
                                                            <?php if ($order['status'] === 'pending'): ?>
                                                            <li>
                                                                <a class="dropdown-item text-success" href="#" 
                                                                   onclick="updateOrderStatus(<?= $order['id'] ?>, 'confirmed')">
                                                                    <i class="fas fa-check"></i> Onayla
                                                                </a>
                                                            </li>
                                                            <?php endif; ?>
                                                            <?php if ($order['status'] === 'confirmed'): ?>
                                                            <li>
                                                                <a class="dropdown-item text-primary" href="#" 
                                                                   onclick="updateOrderStatus(<?= $order['id'] ?>, 'in_progress')">
                                                                    <i class="fas fa-play"></i> İşleme Al
                                                                </a>
                                                            </li>
                                                            <?php endif; ?>
                                                            <?php if ($order['status'] === 'in_progress'): ?>
                                                            <li>
                                                                <a class="dropdown-item text-success" href="#" 
                                                                   onclick="updateOrderStatus(<?= $order['id'] ?>, 'completed')">
                                                                    <i class="fas fa-check-circle"></i> Tamamla
                                                                </a>
                                                            </li>
                                                            <?php endif; ?>
                                                            <li><hr class="dropdown-divider"></li>
                                                            <li>
                                                                <a class="dropdown-item" href="tel:<?= h($order['phone']) ?>">
                                                                    <i class="fas fa-phone text-success"></i> Müşteriyi Ara
                                                                </a>
                                                            </li>
                                                            <li>
                                                                <a class="dropdown-item" href="#" onclick="printOrder(<?= $order['id'] ?>)">
                                                                    <i class="fas fa-print text-info"></i> Yazdır
                                                                </a>
                                                            </li>
                                                            <li><hr class="dropdown-divider"></li>
                                                            <li>
                                                                <a class="dropdown-item text-danger" href="#" 
                                                                   onclick="updateOrderStatus(<?= $order['id'] ?>, 'cancelled')">
                                                                    <i class="fas fa-times"></i> İptal Et
                                                                </a>
                                                            </li>
                                                        </ul>
                                                    </div>
                                                </div>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>

                            <!-- Sayfalama -->
                            <?php if ($pagination['total_pages'] > 1): ?>
                                <nav aria-label="Sayfa navigasyonu" class="mt-4">
                                    <ul class="pagination justify-content-center">
                                        <?php if ($pagination['has_previous']): ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?page=<?= $pagination['current_page'] - 1 ?>&search=<?= urlencode($search) ?>&status=<?= urlencode($status_filter) ?>&date_from=<?= urlencode($date_from) ?>&date_to=<?= urlencode($date_to) ?>">
                                                    <i class="fas fa-chevron-left"></i>
                                                </a>
                                            </li>
                                        <?php endif; ?>

                                        <?php for ($i = max(1, $pagination['current_page'] - 2); $i <= min($pagination['total_pages'], $pagination['current_page'] + 2); $i++): ?>
                                            <li class="page-item <?= $i === $pagination['current_page'] ? 'active' : '' ?>">
                                                <a class="page-link" href="?page=<?= $i ?>&search=<?= urlencode($search) ?>&status=<?= urlencode($status_filter) ?>&date_from=<?= urlencode($date_from) ?>&date_to=<?= urlencode($date_to) ?>">
                                                    <?= $i ?>
                                                </a>
                                            </li>
                                        <?php endfor; ?>

                                        <?php if ($pagination['has_next']): ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?page=<?= $pagination['current_page'] + 1 ?>&search=<?= urlencode($search) ?>&status=<?= urlencode($status_filter) ?>&date_from=<?= urlencode($date_from) ?>&date_to=<?= urlencode($date_to) ?>">
                                                    <i class="fas fa-chevron-right"></i>
                                                </a>
                                            </li>
                                        <?php endif; ?>
                                    </ul>
                                </nav>
                            <?php endif; ?>
                        <?php endif; ?>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="../assets/js/main.js"></script>
    <script>
        function updateOrderStatus(orderId, newStatus) {
            const statusNames = {
                'confirmed': 'Onaylandı',
                'in_progress': 'İşlemde',
                'completed': 'Tamamlandı',
                'cancelled': 'İptal'
            };
            
            if (confirm(`Sipariş durumunu "${statusNames[newStatus]}" olarak değiştirmek istediğinizden emin misiniz?`)) {
                fetch('update_order_status.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: `order_id=${orderId}&status=${newStatus}`
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        AracBakim.showAlert('success', data.message);
                        setTimeout(() => location.reload(), 1500);
                    } else {
                        AracBakim.showAlert('danger', data.message);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    AracBakim.showAlert('danger', 'Bir hata oluştu.');
                });
            }
        }

        function printOrder(orderId) {
            window.open(`print_order.php?id=${orderId}`, '_blank');
        }

        function exportOrders() {
            const searchParams = new URLSearchParams(window.location.search);
            searchParams.set('export', '1');
            window.location.href = 'orders.php?' + searchParams.toString();
        }

        // Auto-refresh every 3 minutes
        setInterval(() => {
            location.reload();
        }, 180000);
    </script>
<?php require_once 'includes/footer.php'; ?>
