<?php
session_start();
require_once '../config/config.php';
require_once '../includes/functions.php';

if (!isLoggedIn()) {
    redirect('login.php');
}

$quote_id = intval($_GET['id'] ?? 0);

if ($quote_id <= 0) {
    redirect('quotes.php');
}

try {
    // Teklif bilgilerini al
    $stmt = $pdo->prepare("
        SELECT q.*, qr.id as request_id, qr.service_type, qr.problem_description,
               c.first_name, c.last_name, c.phone, c.email,
               v.brand, v.model, v.year, v.plate
        FROM quotes q
        JOIN quote_requests qr ON q.quote_request_id = qr.id
        JOIN customers c ON qr.customer_id = c.id
        JOIN vehicles v ON qr.vehicle_id = v.id
        WHERE q.id = ?
    ");
    $stmt->execute([$quote_id]);
    $quote = $stmt->fetch();
    
    if (!$quote) {
        redirect('quotes.php');
    }
    
    // Düzenlenebilir durumda mı kontrol et
    $editable_statuses = ['DRAFT', 'SENT', 'VIEWED'];
    if (!in_array($quote['status'], $editable_statuses)) {
        $_SESSION['error_message'] = 'Bu durumda olan teklif düzenlenemez.';
        redirect('quotes.php');
    }
    
    // Teklif kalemlerini al
    $stmt = $pdo->prepare("
        SELECT qi.*, p.name as product_name, pc.name as category_name
        FROM quote_items qi
        LEFT JOIN products p ON qi.product_id = p.id
        LEFT JOIN product_categories pc ON p.category_id = pc.id
        WHERE qi.quote_id = ?
        ORDER BY qi.sort_order, qi.id
    ");
    $stmt->execute([$quote_id]);
    $quote_items = $stmt->fetchAll();
    
    // Kategorileri al
    $stmt = $pdo->prepare("SELECT * FROM product_categories WHERE is_active = 1 ORDER BY name");
    $stmt->execute();
    $categories = $stmt->fetchAll();
    
    // Ürünleri al
    $stmt = $pdo->prepare("
        SELECT p.*, pc.name as category_name 
        FROM products p 
        LEFT JOIN product_categories pc ON p.category_id = pc.id 
        WHERE p.is_active = 1 
        ORDER BY pc.name, p.name
    ");
    $stmt->execute();
    $products = $stmt->fetchAll();
    
} catch (Exception $e) {
    error_log("Edit quote error: " . $e->getMessage());
    redirect('quotes.php');
}

// Hizmet türleri
$service_types = [
    'periyodik_bakim' => 'Periyodik Bakım',
    'motor_onarim' => 'Motor Onarımı',
    'kaporta_boya' => 'Kaporta & Boya',
    'genel_onarim' => 'Genel Onarım'
];
?>
<?php require_once 'includes/header.php'; ?>

<section class="section">
    <div class="pagetitle">
        <h1>Teklif Düzenle</h1>
        <nav>
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="dashboard.php">Ana Sayfa</a></li>
                <li class="breadcrumb-item"><a href="quotes.php">Teklif Başvuruları</a></li>
                <li class="breadcrumb-item active">Teklif Düzenle</li>
            </ol>
        </nav>
    </div>

    <form id="quoteForm" method="POST" action="process_edit_quote.php">
        <input type="hidden" name="quote_id" value="<?= $quote_id ?>">
        
        <div class="row">
            <!-- Sol Kolon - Müşteri ve Araç Bilgileri -->
            <div class="col-lg-8">
                <!-- Müşteri Bilgileri -->
                <div class="nice-card mb-4">
                    <div class="card-body">
                        <h5 class="card-title text-primary">
                            <i class="fas fa-user me-2"></i>Müşteri Bilgileri
                        </h5>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label">Ad Soyad</label>
                                    <input type="text" class="nice-form-control" 
                                           value="<?= h($quote['first_name'] . ' ' . $quote['last_name']) ?>" readonly>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label">Telefon</label>
                                    <input type="text" class="nice-form-control" 
                                           value="<?= h($quote['phone']) ?>" readonly>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label">E-posta</label>
                                    <input type="text" class="nice-form-control" 
                                           value="<?= h($quote['email']) ?>" readonly>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label">Hizmet Türü</label>
                                    <input type="text" class="nice-form-control" 
                                           value="<?= h($service_types[$quote['service_type']] ?? $quote['service_type']) ?>" readonly>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Araç Bilgileri -->
                <div class="nice-card mb-4">
                    <div class="card-body">
                        <h5 class="card-title text-primary">
                            <i class="fas fa-car me-2"></i>Araç Bilgileri
                        </h5>
                        
                        <div class="row">
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label class="form-label">Marka</label>
                                    <input type="text" class="nice-form-control" 
                                           value="<?= h($quote['brand']) ?>" readonly>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label class="form-label">Model</label>
                                    <input type="text" class="nice-form-control" 
                                           value="<?= h($quote['model']) ?>" readonly>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label class="form-label">Yıl</label>
                                    <input type="text" class="nice-form-control" 
                                           value="<?= h($quote['year']) ?>" readonly>
                                </div>
                            </div>
                            <?php if ($quote['plate']): ?>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label">Plaka</label>
                                    <input type="text" class="nice-form-control" 
                                           value="<?= h($quote['plate']) ?>" readonly>
                                </div>
                            </div>
                            <?php endif; ?>
                        </div>
                        
                        <?php if ($quote['problem_description']): ?>
                        <div class="mb-3">
                            <label class="form-label">Sorun Açıklaması</label>
                            <textarea class="nice-form-control" rows="3" readonly><?= h($quote['problem_description']) ?></textarea>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Teklif Detayları -->
                <div class="nice-card mb-4">
                    <div class="card-body">
                        <h5 class="card-title text-primary">
                            <i class="fas fa-file-invoice me-2"></i>Teklif Detayları
                        </h5>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="discount_rate" class="form-label">İndirim Oranı (%)</label>
                                    <input type="number" class="nice-form-control" id="discount_rate" name="discount_rate" 
                                           value="<?= $quote['discount_percentage'] ?>" min="0" max="100" step="0.01">
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="tax_rate" class="form-label">KDV Oranı (%)</label>
                                    <input type="number" class="nice-form-control" id="tax_rate" name="tax_rate" 
                                           value="<?= $quote['tax_rate'] * 100 ?>" min="0" max="100" step="0.01">
                                </div>
                            </div>
                            <div class="col-12">
                                <div class="mb-3">
                                    <label for="terms_conditions" class="form-label">Şartlar ve Koşullar</label>
                                    <textarea class="nice-form-control" id="terms_conditions" name="terms_conditions" rows="4"><?= h($quote['terms_conditions']) ?></textarea>
                                </div>
                            </div>
                            <div class="col-12">
                                <div class="mb-3">
                                    <label for="internal_notes" class="form-label">İç Notlar</label>
                                    <textarea class="nice-form-control" id="internal_notes" name="internal_notes" rows="3"><?= h($quote['internal_notes']) ?></textarea>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Manuel Ürün Ekleme -->
                <div class="nice-card mb-4" style="background: #f8fafc;">
                    <div class="card-body">
                        <h6 class="card-title text-primary">
                            <i class="fas fa-edit me-2"></i>Manuel Ürün/Hizmet Ekle
                        </h6>
                        
                        <div class="row">
                            <div class="col-md-3 mb-3">
                                <label for="manual_category" class="form-label">Kategori</label>
                                <select class="nice-form-control" id="manual_category">
                                    <option value="">Kategori Seç</option>
                                    <?php foreach ($categories as $category): ?>
                                    <option value="<?= $category['id'] ?>"><?= h($category['name']) ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <div class="col-md-3 mb-3">
                                <label for="manual_name" class="form-label">Ürün/Hizmet Adı</label>
                                <input type="text" class="nice-form-control" id="manual_name" 
                                       placeholder="Örn: Yağ Değişimi">
                            </div>
                            
                            <div class="col-md-2 mb-3">
                                <label for="manual_quantity" class="form-label">Adet</label>
                                <input type="number" class="nice-form-control" id="manual_quantity" 
                                       value="1" min="1">
                            </div>
                            
                            <div class="col-md-2 mb-3">
                                <label for="manual_price" class="form-label">Birim Fiyat (₺)</label>
                                <input type="number" class="nice-form-control" id="manual_price" 
                                       step="0.01" min="0" placeholder="0.00">
                            </div>
                            
                            <div class="col-md-2 mb-3">
                                <label class="form-label">&nbsp;</label>
                                <div class="d-grid">
                                    <button type="button" class="nice-btn nice-btn-primary" onclick="addManualItemToDatabase()">
                                        <i class="fas fa-plus"></i> Ekle & Kaydet
                                    </button>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label for="manual_description" class="form-label">Açıklama</label>
                                <textarea class="nice-form-control" id="manual_description" rows="2" 
                                          placeholder="Ürün/hizmet açıklaması..."></textarea>
                            </div>
                            
                            <div class="col-md-6 mb-3">
                                <label for="manual_type" class="form-label">Ürün Türü</label>
                                <select class="nice-form-control" id="manual_type">
                                    <option value="service">Hizmet</option>
                                    <option value="part">Parça</option>
                                    <option value="package">Paket</option>
                                </select>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Seçili Ürünler -->
                <div class="nice-card">
                    <div class="card-body">
                        <h5 class="card-title text-primary">
                            <i class="fas fa-list me-2"></i>Teklif Kalemleri
                        </h5>
                        
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Ürün/Hizmet</th>
                                        <th>Açıklama</th>
                                        <th>Adet</th>
                                        <th>Birim Fiyat</th>
                                        <th>Toplam</th>
                                        <th>İşlem</th>
                                    </tr>
                                </thead>
                                <tbody id="items_tbody">
                                    <tr id="no_items_row" style="display: none;">
                                        <td colspan="6" class="text-center text-muted py-4">
                                            <i class="fas fa-inbox fa-2x mb-2"></i>
                                            <p>Henüz ürün eklenmedi</p>
                                        </td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Sağ Kolon - Özet ve İşlemler -->
            <div class="col-lg-4">
                <!-- Teklif Özeti -->
                <div class="nice-card mb-4 position-sticky" style="top: 20px;">
                    <div class="card-body">
                        <h5 class="card-title text-primary">
                            <i class="fas fa-calculator me-2"></i>Teklif Özeti
                        </h5>
                        
                        <div class="summary-item">
                            <span>Ara Toplam:</span>
                            <span id="subtotal">₺0.00</span>
                        </div>
                        <div class="summary-item">
                            <span>İndirim:</span>
                            <span id="discount_amount">₺0.00</span>
                        </div>
                        <div class="summary-item">
                            <span>KDV:</span>
                            <span id="tax_amount">₺0.00</span>
                        </div>
                        <hr>
                        <div class="summary-item total">
                            <span><strong>Genel Toplam:</strong></span>
                            <span><strong id="total_amount">₺0.00</strong></span>
                        </div>
                        
                        <div class="d-grid gap-2 mt-4">
                            <button type="submit" class="nice-btn nice-btn-success" id="submitBtn">
                                <i class="fas fa-save"></i> Teklif Güncelle
                            </button>
                            <a href="quotes.php" class="nice-btn nice-btn-outline-secondary">
                                <i class="fas fa-arrow-left"></i> Geri Dön
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </form>
</section>

<style>
.summary-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 0.5rem 0;
    border-bottom: 1px solid #eee;
}

.summary-item.total {
    border-bottom: none;
    border-top: 2px solid #007bff;
    padding-top: 1rem;
    margin-top: 0.5rem;
    font-size: 1.1rem;
}

.summary-item:last-child {
    border-bottom: none;
}
</style>

<script>
let selectedItems = [];
let itemCounter = 0;

// Mevcut teklif kalemlerini yükle
document.addEventListener('DOMContentLoaded', function() {
    const existingItems = <?= json_encode($quote_items) ?>;
    
    existingItems.forEach((item, index) => {
        itemCounter++;
        const quoteItem = {
            id: itemCounter,
            product_id: item.product_id,
            name: item.product_name || item.description,
            description: item.description,
            quantity: parseFloat(item.quantity),
            price: parseFloat(item.unit_price),
            line_total: parseFloat(item.line_total)
        };
        selectedItems.push(quoteItem);
    });
    
    updateItemsTable();
    updateQuoteSummary();
    
    // Event listeners
    document.getElementById('discount_rate').addEventListener('input', updateQuoteSummary);
    document.getElementById('tax_rate').addEventListener('input', updateQuoteSummary);
});

// Manuel ürün ekleme
function addManualItem() {
    const name = document.getElementById('manual_name').value.trim();
    const quantity = parseInt(document.getElementById('manual_quantity').value) || 1;
    const price = parseFloat(document.getElementById('manual_price').value) || 0;
    const description = document.getElementById('manual_description').value.trim();
    
    if (!name || price <= 0) {
        alert('Lütfen ürün adı ve geçerli bir fiyat girin.');
        return;
    }
    
    itemCounter++;
    const total = quantity * price;
    
    const item = {
        id: itemCounter,
        product_id: null,
        name: name,
        description: description || name,
        quantity: quantity,
        price: price,
        line_total: total
    };
    
    selectedItems.push(item);
    updateItemsTable();
    updateQuoteSummary();
    
    // Formu temizle
    document.getElementById('manual_name').value = '';
    document.getElementById('manual_quantity').value = '1';
    document.getElementById('manual_price').value = '';
    document.getElementById('manual_description').value = '';
}

// Ürün tablosunu güncelle
function updateItemsTable() {
    const tbody = document.getElementById('items_tbody');
    const noItemsRow = document.getElementById('no_items_row');
    
    if (selectedItems.length === 0) {
        noItemsRow.style.display = 'table-row';
        return;
    }
    
    noItemsRow.style.display = 'none';
    
    let html = '';
    selectedItems.forEach(item => {
        html += `
            <tr>
                <td>
                    <strong>${escapeHtml(item.name)}</strong>
                </td>
                <td>
                    <small class="text-muted">${escapeHtml(item.description || '')}</small>
                </td>
                <td>
                    <input type="number" class="form-control form-control-sm" style="width: 80px;"
                           value="${item.quantity}" min="1"
                           onchange="updateQuantity(${item.id}, this.value)">
                </td>
                <td>₺${item.price.toFixed(2)}</td>
                <td><strong>₺${item.line_total.toFixed(2)}</strong></td>
                <td>
                    <button type="button" class="btn btn-sm btn-outline-danger" onclick="removeItem(${item.id})">
                        <i class="fas fa-trash"></i>
                    </button>
                </td>
            </tr>
        `;
    });
    
    tbody.innerHTML = html + noItemsRow.outerHTML;
}

// Miktar güncelle
function updateQuantity(itemId, newQuantity) {
    const item = selectedItems.find(i => i.id === itemId);
    if (item) {
        item.quantity = parseInt(newQuantity) || 1;
        item.line_total = item.quantity * item.price;
        updateItemsTable();
        updateQuoteSummary();
    }
}

// Ürün sil
function removeItem(itemId) {
    selectedItems = selectedItems.filter(item => item.id !== itemId);
    updateItemsTable();
    updateQuoteSummary();
}

// Teklif özetini güncelle
function updateQuoteSummary() {
    const discountRate = parseFloat(document.getElementById('discount_rate').value) || 0;
    const taxRate = parseFloat(document.getElementById('tax_rate').value) || 18;
    
    let subtotal = 0;
    selectedItems.forEach(item => {
        subtotal += item.line_total;
    });
    
    const discountAmount = subtotal * (discountRate / 100);
    const taxableAmount = subtotal - discountAmount;
    const taxAmount = taxableAmount * (taxRate / 100);
    const total = taxableAmount + taxAmount;
    
    document.getElementById('subtotal').textContent = `₺${subtotal.toFixed(2)}`;
    document.getElementById('discount_amount').textContent = `₺${discountAmount.toFixed(2)}`;
    document.getElementById('tax_amount').textContent = `₺${taxAmount.toFixed(2)}`;
    document.getElementById('total_amount').textContent = `₺${total.toFixed(2)}`;
}

// Manuel ürün ekleme ve veritabanına kaydetme
function addManualItemToDatabase() {
    const categoryId = document.getElementById('manual_category').value;
    const name = document.getElementById('manual_name').value.trim();
    const quantity = parseInt(document.getElementById('manual_quantity').value) || 1;
    const price = parseFloat(document.getElementById('manual_price').value) || 0;
    const description = document.getElementById('manual_description').value.trim();
    const type = document.getElementById('manual_type').value;
    
    if (!name || price <= 0) {
        alert('Lütfen ürün adı ve geçerli bir fiyat girin.');
        return;
    }
    
    // AJAX ile ürünü veritabanına kaydet
    const formData = new FormData();
    formData.append('action', 'add_product');
    formData.append('category_id', categoryId || '');
    formData.append('name', name);
    formData.append('description', description);
    formData.append('type', type);
    formData.append('price', price);
    
    fetch('add_product_ajax.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Ürün başarıyla kaydedildi, şimdi teklife ekle
            itemCounter++;
            const newItem = {
                id: 'manual_' + itemCounter,
                product_id: data.product_id,
                name: name,
                description: description,
                quantity: quantity,
                price: price,
                line_total: quantity * price,
                is_manual: true
            };
            
            selectedItems.push(newItem);
            updateItemsTable();
            updateQuoteSummary();
            clearManualForm();
            showSuccessMessage('Ürün başarıyla eklendi ve kataloga kaydedildi!');
        } else {
            alert('Hata: ' + data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Bir hata oluştu: ' + error.message);
    });
}

// Manuel form temizleme
function clearManualForm() {
    document.getElementById('manual_name').value = '';
    document.getElementById('manual_quantity').value = '1';
    document.getElementById('manual_price').value = '';
    document.getElementById('manual_description').value = '';
    document.getElementById('manual_type').value = 'service';
}

// Başarı mesajı göster
function showSuccessMessage(message) {
    const alertDiv = document.createElement('div');
    alertDiv.className = 'alert alert-success alert-dismissible fade show';
    alertDiv.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    const container = document.querySelector('.container-fluid');
    container.insertBefore(alertDiv, container.firstChild);
    
    // 3 saniye sonra otomatik kapat
    setTimeout(() => {
        if (alertDiv.parentNode) {
            alertDiv.remove();
        }
    }, 3000);
}

// Kategori değiştiğinde ürünleri yükle
function loadCategoryProducts() {
    const categoryId = document.getElementById('category_select').value;
    const productSelect = document.getElementById('product_select');
    
    if (!categoryId) {
        productSelect.innerHTML = '<option value="">Önce kategori seçin</option>';
        return;
    }
    
    productSelect.innerHTML = '<option value="">Yükleniyor...</option>';
    
    fetch(`get_products.php?category_id=${categoryId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                productSelect.innerHTML = '<option value="">Ürün seçin</option>';
                data.products.forEach(product => {
                    const option = document.createElement('option');
                    option.value = product.id;
                    option.textContent = `${product.name} - ₺${parseFloat(product.price).toFixed(2)}`;
                    option.dataset.name = product.name;
                    option.dataset.price = product.price;
                    option.dataset.description = product.description || '';
                    productSelect.appendChild(option);
                });
            } else {
                productSelect.innerHTML = '<option value="">Ürün bulunamadı</option>';
            }
        })
        .catch(error => {
            console.error('Error:', error);
            productSelect.innerHTML = '<option value="">Hata oluştu</option>';
        });
}

// HTML escape
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Form gönderimi
document.getElementById('quoteForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    if (selectedItems.length === 0) {
        alert('Lütfen en az bir ürün/hizmet ekleyin.');
        return;
    }
    
    const formData = new FormData(this);
    
    // Seçili ürünleri ekle
    selectedItems.forEach((item, index) => {
        formData.append(`items[${index}][product_id]`, item.product_id || '');
        formData.append(`items[${index}][name]`, item.name);
        formData.append(`items[${index}][description]`, item.description);
        formData.append(`items[${index}][quantity]`, item.quantity);
        formData.append(`items[${index}][unit_price]`, item.price);
        formData.append(`items[${index}][line_total]`, item.line_total);
    });
    
    const submitBtn = document.getElementById('submitBtn');
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Güncelleniyor...';
    
    fetch('process_edit_quote.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert('Teklif başarıyla güncellendi!');
            window.location.href = 'quotes.php';
        } else {
            alert('Hata: ' + data.message);
            submitBtn.disabled = false;
            submitBtn.innerHTML = '<i class="fas fa-save"></i> Teklif Güncelle';
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Bir hata oluştu. Lütfen tekrar deneyin.');
        submitBtn.disabled = false;
        submitBtn.innerHTML = '<i class="fas fa-save"></i> Teklif Güncelle';
    });
});
</script>

<?php require_once 'includes/footer.php'; ?>
