<?php
session_start();
require_once '../config/config.php';
require_once '../includes/functions.php';

$pageTitle = 'Müşteriler';
$currentPage = 'customers';

// Sayfalama
$page = intval($_GET['page'] ?? 1);
$limit = 20;
$offset = ($page - 1) * $limit;

// Filtreleme
$search = trim($_GET['search'] ?? '');
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';

// Müşterileri al
$where_conditions = ['1=1'];
$params = [];

if ($search) {
    $where_conditions[] = "(c.first_name LIKE ? OR c.last_name LIKE ? OR c.phone LIKE ? OR c.email LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

if ($date_from) {
    $where_conditions[] = "DATE(c.created_at) >= ?";
    $params[] = $date_from;
}

if ($date_to) {
    $where_conditions[] = "DATE(c.created_at) <= ?";
    $params[] = $date_to;
}

$where_clause = implode(' AND ', $where_conditions);

try {
    // Toplam kayıt sayısı
    $count_sql = "SELECT COUNT(*) as total FROM customers c WHERE $where_clause";
    $stmt = $pdo->prepare($count_sql);
    $stmt->execute($params);
    $total_records = $stmt->fetch()['total'];

    // Müşterileri getir
    $sql = "
        SELECT c.*,
               COUNT(DISTINCT qr.id) as quote_count,
               COUNT(DISTINCT o.id) as order_count,
               COALESCE(SUM(o.total_amount), 0) as total_spent,
               MAX(qr.created_at) as last_quote_date
        FROM customers c
        LEFT JOIN quote_requests qr ON c.id = qr.customer_id
        LEFT JOIN orders o ON c.id = o.customer_id AND o.status = 'completed'
        WHERE $where_clause
        GROUP BY c.id
        ORDER BY c.created_at DESC
        LIMIT $limit OFFSET $offset
    ";

    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $customers = $stmt->fetchAll();

    $pagination = calculatePagination($total_records, $page, $limit);
    
} catch (Exception $e) {
    error_log("Customers page error: " . $e->getMessage());
    $customers = [];
    $pagination = ['total_pages' => 0, 'current_page' => 1];
}
?>
<?php require_once 'includes/header.php'; ?>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <?php include 'includes/sidebar.php'; ?>

            <!-- Ana İçerik -->
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2"><i class="fas fa-users"></i> Müşteri Yönetimi</h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <div class="btn-group me-2">
                            <button type="button" class="btn btn-sm btn-outline-secondary" onclick="exportCustomers()">
                                <i class="fas fa-download"></i> Dışa Aktar
                            </button>
                        </div>
                    </div>
                </div>

                <!-- İstatistikler -->
                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="stats-card">
                            <div class="d-flex align-items-center">
                                <div class="stats-icon bg-primary">
                                    <i class="fas fa-users"></i>
                                </div>
                                <div class="ms-3">
                                    <div class="stats-number"><?= $total_records ?></div>
                                    <div class="stats-label">Toplam Müşteri</div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stats-card">
                            <div class="d-flex align-items-center">
                                <div class="stats-icon bg-success">
                                    <i class="fas fa-user-check"></i>
                                </div>
                                <div class="ms-3">
                                    <?php
                                    $stmt = $pdo->query("SELECT COUNT(*) as count FROM customers WHERE phone_verified = 1");
                                    $verified_count = $stmt->fetch()['count'];
                                    ?>
                                    <div class="stats-number"><?= $verified_count ?></div>
                                    <div class="stats-label">Doğrulanmış</div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stats-card">
                            <div class="d-flex align-items-center">
                                <div class="stats-icon bg-info">
                                    <i class="fas fa-calendar-day"></i>
                                </div>
                                <div class="ms-3">
                                    <?php
                                    $stmt = $pdo->query("SELECT COUNT(*) as count FROM customers WHERE DATE(created_at) = CURDATE()");
                                    $today_count = $stmt->fetch()['count'];
                                    ?>
                                    <div class="stats-number"><?= $today_count ?></div>
                                    <div class="stats-label">Bugün Kayıt</div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stats-card">
                            <div class="d-flex align-items-center">
                                <div class="stats-icon bg-warning">
                                    <i class="fas fa-calendar-week"></i>
                                </div>
                                <div class="ms-3">
                                    <?php
                                    $stmt = $pdo->query("SELECT COUNT(*) as count FROM customers WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)");
                                    $week_count = $stmt->fetch()['count'];
                                    ?>
                                    <div class="stats-number"><?= $week_count ?></div>
                                    <div class="stats-label">Bu Hafta</div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Filtreler -->
                <div class="card mb-4">
                    <div class="card-body">
                        <form method="GET" class="row g-3">
                            <div class="col-md-4">
                                <label for="search" class="form-label">Arama</label>
                                <input type="text" class="form-control" id="search" name="search" 
                                       value="<?= h($search) ?>" placeholder="Ad, soyad, telefon veya e-posta">
                            </div>
                            <div class="col-md-3">
                                <label for="date_from" class="form-label">Başlangıç Tarihi</label>
                                <input type="date" class="form-control" id="date_from" name="date_from" value="<?= h($date_from) ?>">
                            </div>
                            <div class="col-md-3">
                                <label for="date_to" class="form-label">Bitiş Tarihi</label>
                                <input type="date" class="form-control" id="date_to" name="date_to" value="<?= h($date_to) ?>">
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">&nbsp;</label>
                                <div class="d-grid">
                                    <button type="submit" class="btn btn-primary">
                                        <i class="fas fa-search"></i> Filtrele
                                    </button>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Müşteri Listesi -->
                <div class="card">
                    <div class="card-header">
                        <h5><i class="fas fa-list"></i> Müşteri Listesi (<?= $total_records ?> kayıt)</h5>
                    </div>
                    <div class="card-body">
                        <?php if (empty($customers)): ?>
                            <div class="text-center py-5">
                                <i class="fas fa-users fa-3x text-muted mb-3"></i>
                                <p class="text-muted">Henüz müşteri kaydı bulunmuyor.</p>
                            </div>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>Müşteri</th>
                                            <th>İletişim</th>
                                            <th>Kayıt Tarihi</th>
                                            <th>Teklif Sayısı</th>
                                            <th>Sipariş Sayısı</th>
                                            <th>Toplam Harcama</th>
                                            <th>Son Aktivite</th>
                                            <th>İşlemler</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($customers as $customer): ?>
                                        <tr>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <div class="avatar-circle me-2">
                                                        <?= strtoupper(substr($customer['first_name'], 0, 1) . substr($customer['last_name'], 0, 1)) ?>
                                                    </div>
                                                    <div>
                                                        <strong><?= h($customer['first_name'] . ' ' . $customer['last_name']) ?></strong>
                                                        <?php if ($customer['phone_verified']): ?>
                                                            <br><small class="text-success"><i class="fas fa-check-circle"></i> Doğrulanmış</small>
                                                        <?php else: ?>
                                                            <br><small class="text-warning"><i class="fas fa-exclamation-triangle"></i> Doğrulanmamış</small>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                            </td>
                                            <td>
                                                <div>
                                                    <i class="fas fa-phone text-primary"></i> 
                                                    <a href="tel:<?= h($customer['phone']) ?>" class="text-decoration-none">
                                                        <?= h($customer['phone']) ?>
                                                    </a>
                                                </div>
                                                <?php if ($customer['email']): ?>
                                                <div class="mt-1">
                                                    <i class="fas fa-envelope text-info"></i> 
                                                    <a href="mailto:<?= h($customer['email']) ?>" class="text-decoration-none">
                                                        <?= h($customer['email']) ?>
                                                    </a>
                                                </div>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <small><?= formatDate($customer['created_at'], 'd.m.Y') ?></small>
                                            </td>
                                            <td>
                                                <?php if ($customer['quote_count'] > 0): ?>
                                                    <span class="badge bg-primary"><?= $customer['quote_count'] ?></span>
                                                <?php else: ?>
                                                    <span class="text-muted">0</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <?php if ($customer['order_count'] > 0): ?>
                                                    <span class="badge bg-success"><?= $customer['order_count'] ?></span>
                                                <?php else: ?>
                                                    <span class="text-muted">0</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <?php if ($customer['total_spent'] > 0): ?>
                                                    <strong class="text-success"><?= formatMoney($customer['total_spent']) ?></strong>
                                                <?php else: ?>
                                                    <span class="text-muted">0,00 ₺</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <?php if ($customer['last_quote_date']): ?>
                                                    <small><?= formatDate($customer['last_quote_date'], 'd.m.Y') ?></small>
                                                <?php else: ?>
                                                    <span class="text-muted">-</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <div class="btn-group btn-group-sm">
                                                    <a href="customer_detail.php?id=<?= $customer['id'] ?>" 
                                                       class="btn btn-outline-primary" title="Detay">
                                                        <i class="fas fa-eye"></i>
                                                    </a>
                                                    <div class="btn-group btn-group-sm">
                                                        <button type="button" class="btn btn-outline-secondary dropdown-toggle" 
                                                                data-bs-toggle="dropdown">
                                                            <i class="fas fa-ellipsis-v"></i>
                                                        </button>
                                                        <ul class="dropdown-menu">
                                                            <li>
                                                                <a class="dropdown-item" href="tel:<?= h($customer['phone']) ?>">
                                                                    <i class="fas fa-phone text-success"></i> Ara
                                                                </a>
                                                            </li>
                                                            <?php if ($customer['email']): ?>
                                                            <li>
                                                                <a class="dropdown-item" href="mailto:<?= h($customer['email']) ?>">
                                                                    <i class="fas fa-envelope text-info"></i> E-posta Gönder
                                                                </a>
                                                            </li>
                                                            <?php endif; ?>
                                                            <li><hr class="dropdown-divider"></li>
                                                            <li>
                                                                <a class="dropdown-item" href="create_quote.php?customer_id=<?= $customer['id'] ?>">
                                                                    <i class="fas fa-plus text-primary"></i> Yeni Teklif
                                                                </a>
                                                            </li>
                                                            <li>
                                                                <a class="dropdown-item text-danger" href="#" 
                                                                   onclick="deleteCustomer(<?= $customer['id'] ?>, '<?= h($customer['first_name'] . ' ' . $customer['last_name']) ?>')">
                                                                    <i class="fas fa-trash"></i> Sil
                                                                </a>
                                                            </li>
                                                        </ul>
                                                    </div>
                                                </div>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>

                            <!-- Sayfalama -->
                            <?php if ($pagination['total_pages'] > 1): ?>
                                <nav aria-label="Sayfa navigasyonu" class="mt-4">
                                    <ul class="pagination justify-content-center">
                                        <?php if ($pagination['has_previous']): ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?page=<?= $pagination['current_page'] - 1 ?>&search=<?= urlencode($search) ?>&date_from=<?= urlencode($date_from) ?>&date_to=<?= urlencode($date_to) ?>">
                                                    <i class="fas fa-chevron-left"></i>
                                                </a>
                                            </li>
                                        <?php endif; ?>

                                        <?php for ($i = max(1, $pagination['current_page'] - 2); $i <= min($pagination['total_pages'], $pagination['current_page'] + 2); $i++): ?>
                                            <li class="page-item <?= $i === $pagination['current_page'] ? 'active' : '' ?>">
                                                <a class="page-link" href="?page=<?= $i ?>&search=<?= urlencode($search) ?>&date_from=<?= urlencode($date_from) ?>&date_to=<?= urlencode($date_to) ?>">
                                                    <?= $i ?>
                                                </a>
                                            </li>
                                        <?php endfor; ?>

                                        <?php if ($pagination['has_next']): ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?page=<?= $pagination['current_page'] + 1 ?>&search=<?= urlencode($search) ?>&date_from=<?= urlencode($date_from) ?>&date_to=<?= urlencode($date_to) ?>">
                                                    <i class="fas fa-chevron-right"></i>
                                                </a>
                                            </li>
                                        <?php endif; ?>
                                    </ul>
                                </nav>
                            <?php endif; ?>
                        <?php endif; ?>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="../assets/js/main.js"></script>
    <script>
        function deleteCustomer(id, name) {
            if (confirm(`"${name}" müşterisini silmek istediğinizden emin misiniz?\n\nBu işlem geri alınamaz ve müşteriye ait tüm veriler silinecektir.`)) {
                fetch('delete_customer.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: `id=${id}`
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        AracBakim.showAlert('success', data.message);
                        setTimeout(() => location.reload(), 1500);
                    } else {
                        AracBakim.showAlert('danger', data.message);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    AracBakim.showAlert('danger', 'Bir hata oluştu.');
                });
            }
        }

        function exportCustomers() {
            const searchParams = new URLSearchParams(window.location.search);
            searchParams.set('export', '1');
            window.location.href = 'customers.php?' + searchParams.toString();
        }
    </script>

    <style>
        .avatar-circle {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--primary-dark) 100%);
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
            font-size: 0.8rem;
        }
    </style>
<?php require_once 'includes/footer.php'; ?>
