<?php
session_start();
require_once '../config/config.php';
require_once '../includes/functions.php';

$pageTitle = 'Teklif Hazırla';
$currentPage = 'quotes';

if (!isLoggedIn()) {
    redirect('login.php');
}

$request_id = intval($_GET['request_id'] ?? 0);

if (!$request_id) {
    addFlashMessage('danger', 'Geçersiz başvuru ID.');
    redirect('quotes.php');
}

try {
    // Başvuru bilgilerini al
    $stmt = $pdo->prepare("
        SELECT qr.*, c.first_name, c.last_name, c.phone, c.email, c.contact_address,
               v.brand, v.model, v.year, v.fuel_type, v.transmission, v.plate, v.mileage, v.chassis_number
        FROM quote_requests qr
        JOIN customers c ON qr.customer_id = c.id
        JOIN vehicles v ON qr.vehicle_id = v.id
        WHERE qr.id = ?
    ");
    $stmt->execute([$request_id]);
    $request = $stmt->fetch();
    
    if (!$request) {
        addFlashMessage('danger', 'Başvuru bulunamadı.');
        redirect('quotes.php');
    }
    
    // Varsa mevcut teklifi kontrol et
    $stmt = $pdo->prepare("SELECT id FROM quotes WHERE quote_request_id = ?");
    $stmt->execute([$request_id]);
    $existing_quote = $stmt->fetch();
    
    if ($existing_quote) {
        addFlashMessage('info', 'Bu başvuru için zaten teklif hazırlanmış.');
        redirect('view_quote.php?id=' . $request_id);
    }
    
    // Kategorileri al
    $stmt = $pdo->query("SELECT * FROM product_categories WHERE is_active = 1 ORDER BY name");
    $categories = $stmt->fetchAll();
    
    // Ürün/hizmetleri al
    $stmt = $pdo->prepare("
        SELECT p.*, pc.name as category_name
        FROM products p
        LEFT JOIN product_categories pc ON p.category_id = pc.id
        WHERE p.is_active = 1
        ORDER BY pc.name, p.name
    ");
    $stmt->execute();
    $products = $stmt->fetchAll();
    
    // Hizmet türü çevirisi
    $service_types = [
        'periyodik_bakim' => 'Periyodik Bakım',
        'motor_onarim' => 'Motor Onarımı',
        'kaporta_boya' => 'Kaporta & Boya',
        'genel_onarim' => 'Genel Onarım'
    ];
    
} catch (Exception $e) {
    error_log("Create quote error: " . $e->getMessage());
    addFlashMessage('danger', 'Bir hata oluştu.');
    redirect('quotes.php');
}
?>
<?php require_once 'includes/header.php'; ?>

<section class="section">
    <div class="pagetitle">
        <h1>Teklif Hazırla</h1>
        <nav>
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="dashboard.php">Ana Sayfa</a></li>
                <li class="breadcrumb-item"><a href="quotes.php">Teklif Başvuruları</a></li>
                <li class="breadcrumb-item active">Teklif Hazırla</li>
            </ol>
        </nav>
    </div>

    <div class="row">
        <div class="col-lg-8">
            <!-- Quote Form -->
            <div class="nice-card">
                <div class="card-body">
                    <h5 class="card-title">
                        <i class="fas fa-calculator me-2"></i>Teklif Detayları
                    </h5>
                    
                    <form id="quoteForm" action="process_create_quote.php" method="POST">
                        <input type="hidden" name="request_id" value="<?= $request_id ?>">
                        
                        <!-- Teklif Bilgileri -->
                        <div class="row mb-4">
                            <div class="col-12">
                                <h6 class="text-primary border-bottom pb-2 mb-3">
                                    <i class="fas fa-file-invoice me-2"></i>Teklif Bilgileri
                                </h6>
                            </div>
                            
                            <div class="col-md-6 mb-3">
                                <label for="quote_title" class="form-label">Teklif Başlığı</label>
                                <input type="text" class="nice-form-control" id="quote_title" name="quote_title" 
                                       value="<?= h($request['brand'] . ' ' . $request['model'] . ' - ' . ($service_types[$request['service_type']] ?? $request['service_type'])) ?>" required>
                            </div>
                            
                            <div class="col-md-6 mb-3">
                                <label for="valid_days" class="form-label">Geçerlilik Süresi (Gün)</label>
                                <input type="number" class="nice-form-control" id="valid_days" name="valid_days" 
                                       value="7" min="1" max="30" required>
                            </div>
                            
                            <div class="col-md-6 mb-3">
                                <label for="tax_rate" class="form-label">KDV Oranı (%)</label>
                                <input type="number" class="nice-form-control" id="tax_rate" name="tax_rate" 
                                       value="18" min="0" max="100" step="0.01" required>
                            </div>
                            
                            <div class="col-md-6 mb-3">
                                <label for="discount_rate" class="form-label">İndirim Oranı (%)</label>
                                <input type="number" class="nice-form-control" id="discount_rate" name="discount_rate" 
                                       value="0" min="0" max="100" step="0.01">
                            </div>
                            
                            <div class="col-12 mb-3">
                                <label for="notes" class="form-label">Teklif Notları</label>
                                <textarea class="nice-form-control" id="notes" name="notes" rows="3" 
                                          placeholder="Teklif ile ilgili özel notlar..."></textarea>
                            </div>
                        </div>

                        <!-- Ürün/Hizmet Ekleme -->
                        <div class="row mb-4">
                            <div class="col-12">
                                <h6 class="text-primary border-bottom pb-2 mb-3">
                                    <i class="fas fa-plus-circle me-2"></i>Ürün/Hizmet Ekle
                                </h6>
                            </div>
                            
                            <!-- Manuel Ürün Ekleme -->
                            <div class="col-12 mb-4">
                                <div class="nice-card" style="background: #f8fafc;">
                                    <div class="card-body">
                                        <h6 class="card-title text-primary">
                                            <i class="fas fa-edit me-2"></i>Manuel Ürün/Hizmet Ekle
                                        </h6>
                                        
                                        <div class="row">
                                            <div class="col-md-3 mb-3">
                                                <label for="manual_category" class="form-label">Kategori</label>
                                                <select class="nice-form-control" id="manual_category">
                                                    <option value="">Kategori Seç</option>
                                                    <?php foreach ($categories as $category): ?>
                                                    <option value="<?= $category['id'] ?>"><?= h($category['name']) ?></option>
                                                    <?php endforeach; ?>
                                                </select>
                                            </div>
                                            
                                            <div class="col-md-3 mb-3">
                                                <label for="manual_name" class="form-label">Ürün/Hizmet Adı</label>
                                                <input type="text" class="nice-form-control" id="manual_name" 
                                                       placeholder="Örn: Yağ Değişimi">
                                            </div>
                                            
                                            <div class="col-md-2 mb-3">
                                                <label for="manual_quantity" class="form-label">Adet</label>
                                                <input type="number" class="nice-form-control" id="manual_quantity" 
                                                       value="1" min="1">
                                            </div>
                                            
                                            <div class="col-md-2 mb-3">
                                                <label for="manual_price" class="form-label">Birim Fiyat (₺)</label>
                                                <input type="number" class="nice-form-control" id="manual_price" 
                                                       step="0.01" min="0" placeholder="0.00">
                                            </div>
                                            
                                            <div class="col-md-2 mb-3">
                                                <label class="form-label">&nbsp;</label>
                                                <div class="d-grid">
                                                    <button type="button" class="nice-btn nice-btn-primary" onclick="addManualItemToDatabase()">
                                                        <i class="fas fa-plus"></i> Ekle & Kaydet
                                                    </button>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <div class="row">
                                            <div class="col-md-6 mb-3">
                                                <label for="manual_description" class="form-label">Açıklama</label>
                                                <textarea class="nice-form-control" id="manual_description" rows="2" 
                                                          placeholder="Ürün/hizmet açıklaması..."></textarea>
                                            </div>
                                            
                                            <div class="col-md-6 mb-3">
                                                <label for="manual_type" class="form-label">Ürün Türü</label>
                                                <select class="nice-form-control" id="manual_type">
                                                    <option value="service">Hizmet</option>
                                                    <option value="part">Parça</option>
                                                    <option value="package">Paket</option>
                                                </select>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Kategori Bazlı Ürün Seçimi -->
                            <div class="col-12 mb-4">
                                <div class="nice-card" style="background: #f0f9ff;">
                                    <div class="card-body">
                                        <h6 class="card-title text-primary">
                                            <i class="fas fa-search me-2"></i>Mevcut Ürünlerden Seç
                                        </h6>
                                        
                                        <div class="row">
                                            <div class="col-md-6 mb-3">
                                                <label for="category_filter" class="form-label">Kategori</label>
                                                <select class="nice-form-control" id="category_filter" onchange="filterProducts()">
                                                    <option value="">Tüm Kategoriler</option>
                                                    <?php foreach ($categories as $category): ?>
                                                        <option value="<?= $category['id'] ?>"><?= h($category['name']) ?></option>
                                                    <?php endforeach; ?>
                                                </select>
                                            </div>
                                            
                                            <div class="col-md-6 mb-3">
                                                <label for="product_search" class="form-label">Ürün Ara</label>
                                                <input type="text" class="nice-form-control" id="product_search" 
                                                       placeholder="Ürün adı ile ara..." onkeyup="filterProducts()">
                                            </div>
                                        </div>
                                        
                                        <div id="products_list" class="row">
                                            <?php foreach ($products as $product): ?>
                                            <div class="col-md-6 col-lg-4 mb-3 product-item" 
                                                 data-category="<?= $product['category_id'] ?>" 
                                                 data-name="<?= strtolower($product['name']) ?>">
                                                <div class="card h-100" style="cursor: pointer; transition: all 0.3s ease;" 
                                                     onclick="addProductToQuote(<?= $product['id'] ?>, '<?= h($product['name']) ?>', <?= $product['price'] ?>, '<?= h($product['description']) ?>')">
                                                    <div class="card-body p-3">
                                                        <h6 class="card-title text-truncate" title="<?= h($product['name']) ?>">
                                                            <?= h($product['name']) ?>
                                                        </h6>
                                                        <p class="card-text small text-muted mb-2">
                                                            <?= h($product['category_name'] ?? 'Kategori Yok') ?>
                                                        </p>
                                                        <div class="d-flex justify-content-between align-items-center">
                                                            <span class="fw-bold text-primary"><?= formatMoney($product['price']) ?></span>
                                                            <i class="fas fa-plus-circle text-success"></i>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Seçili Ürünler -->
                        <div class="row mb-4">
                            <div class="col-12">
                                <h6 class="text-primary border-bottom pb-2 mb-3">
                                    <i class="fas fa-list me-2"></i>Seçili Ürün/Hizmetler
                                </h6>
                            </div>
                            
                            <div class="col-12">
                                <div id="selected_items" class="table-responsive">
                                    <table class="nice-table">
                                        <thead>
                                            <tr>
                                                <th>Ürün/Hizmet</th>
                                                <th>Açıklama</th>
                                                <th>Adet</th>
                                                <th>Birim Fiyat</th>
                                                <th>Toplam</th>
                                                <th>İşlem</th>
                                            </tr>
                                        </thead>
                                        <tbody id="items_tbody">
                                            <tr id="no_items_row">
                                                <td colspan="6" class="text-center text-muted py-4">
                                                    <i class="fas fa-inbox fa-2x mb-2"></i>
                                                    <br>Henüz ürün/hizmet eklenmedi
                                                </td>
                                            </tr>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>

                        <!-- Form Buttons -->
                        <div class="text-end">
                            <a href="quotes.php" class="nice-btn nice-btn-outline-secondary me-2">
                                <i class="fas fa-times"></i> İptal
                            </a>
                            <button type="submit" class="nice-btn nice-btn-success" id="submitBtn">
                                <i class="fas fa-paper-plane"></i> Teklif Gönder
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <div class="col-lg-4">
            <!-- Başvuru Bilgileri -->
            <div class="nice-card mb-4">
                <div class="card-body">
                    <h5 class="card-title">
                        <i class="fas fa-info-circle me-2"></i>Başvuru Bilgileri
                    </h5>
                    
                    <div class="mb-3">
                        <h6 class="text-primary">Müşteri Bilgileri</h6>
                        <p class="mb-1"><strong><?= h($request['first_name'] . ' ' . $request['last_name']) ?></strong></p>
                        <p class="mb-1 text-muted"><i class="fas fa-phone me-1"></i> <?= h($request['phone']) ?></p>
                        <?php if ($request['email']): ?>
                        <p class="mb-1 text-muted"><i class="fas fa-envelope me-1"></i> <?= h($request['email']) ?></p>
                        <?php endif; ?>
                        <?php if ($request['contact_address']): ?>
                        <p class="mb-0 text-muted"><i class="fas fa-map-marker-alt me-1"></i> <?= h($request['contact_address']) ?></p>
                        <?php endif; ?>
                    </div>
                    
                    <div class="mb-3">
                        <h6 class="text-primary">Araç Bilgileri</h6>
                        <p class="mb-1"><strong><?= h($request['brand'] . ' ' . $request['model'] . ' (' . $request['year'] . ')') ?></strong></p>
                        <?php if ($request['plate']): ?>
                        <p class="mb-1 text-muted"><i class="fas fa-car me-1"></i> <?= h($request['plate']) ?></p>
                        <?php endif; ?>
                        <?php if ($request['fuel_type']): ?>
                        <p class="mb-1 text-muted"><i class="fas fa-gas-pump me-1"></i> <?= ucfirst($request['fuel_type']) ?></p>
                        <?php endif; ?>
                        <?php if ($request['transmission']): ?>
                        <p class="mb-1 text-muted"><i class="fas fa-cogs me-1"></i> <?= ucfirst($request['transmission']) ?></p>
                        <?php endif; ?>
                        <?php if ($request['mileage']): ?>
                        <p class="mb-0 text-muted"><i class="fas fa-tachometer-alt me-1"></i> <?= number_format($request['mileage']) ?> km</p>
                        <?php endif; ?>
                    </div>
                    
                    <div class="mb-3">
                        <h6 class="text-primary">Hizmet Detayları</h6>
                        <p class="mb-1"><strong><?= h($service_types[$request['service_type']] ?? $request['service_type']) ?></strong></p>
                        <?php if ($request['problem_description']): ?>
                        <p class="mb-1 text-muted"><?= h($request['problem_description']) ?></p>
                        <?php endif; ?>
                        <?php if ($request['preferred_date']): ?>
                        <p class="mb-1 text-muted"><i class="fas fa-calendar me-1"></i> <?= formatDate($request['preferred_date'], 'd.m.Y') ?></p>
                        <?php endif; ?>
                        <?php if ($request['preferred_time']): ?>
                        <p class="mb-0 text-muted"><i class="fas fa-clock me-1"></i> <?= h($request['preferred_time']) ?></p>
                        <?php endif; ?>
                    </div>
                    
                    <div class="text-muted small">
                        <i class="fas fa-calendar-alt me-1"></i>
                        Başvuru Tarihi: <?= formatDate($request['created_at'], 'd.m.Y H:i') ?>
                    </div>
                </div>
            </div>

            <!-- Teklif Özeti -->
            <div class="nice-card">
                <div class="card-body">
                    <h5 class="card-title">
                        <i class="fas fa-calculator me-2"></i>Teklif Özeti
                    </h5>
                    
                    <div id="quote_summary">
                        <div class="d-flex justify-content-between mb-2">
                            <span>Ara Toplam:</span>
                            <span id="subtotal">₺0.00</span>
                        </div>
                        <div class="d-flex justify-content-between mb-2">
                            <span>İndirim:</span>
                            <span id="discount_amount">₺0.00</span>
                        </div>
                        <div class="d-flex justify-content-between mb-2">
                            <span>KDV:</span>
                            <span id="tax_amount">₺0.00</span>
                        </div>
                        <hr>
                        <div class="d-flex justify-content-between fw-bold text-primary">
                            <span>Toplam:</span>
                            <span id="total_amount">₺0.00</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>

<script>
let selectedItems = [];
let itemCounter = 0;

// Manuel ürün ekleme (sadece teklife ekler)
function addManualItem() {
    const name = document.getElementById('manual_name').value.trim();
    const quantity = parseInt(document.getElementById('manual_quantity').value) || 1;
    const price = parseFloat(document.getElementById('manual_price').value) || 0;
    const description = document.getElementById('manual_description').value.trim();
    
    if (!name || price <= 0) {
        alert('Lütfen ürün adı ve geçerli bir fiyat girin.');
        return;
    }
    
    // Manuel ürün için product_id null olacak
    itemCounter++;
    const total = quantity * price;
    
    const item = {
        id: itemCounter,
        product_id: null, // Manuel ürün
        name: name,
        description: description || name,
        quantity: quantity,
        price: price,
        line_total: total
    };
    
    selectedItems.push(item);
    updateItemsTable();
    updateQuoteSummary();
    
    // Formu temizle
    clearManualForm();
}

// Manuel ürün ekleme ve veritabanına kaydetme
function addManualItemToDatabase() {
    const categoryId = document.getElementById('manual_category').value;
    const name = document.getElementById('manual_name').value.trim();
    const quantity = parseInt(document.getElementById('manual_quantity').value) || 1;
    const price = parseFloat(document.getElementById('manual_price').value) || 0;
    const description = document.getElementById('manual_description').value.trim();
    const type = document.getElementById('manual_type').value;
    
    if (!name || price <= 0) {
        alert('Lütfen ürün adı ve geçerli bir fiyat girin.');
        return;
    }
    
    // AJAX ile ürünü veritabanına kaydet
    const formData = new FormData();
    formData.append('action', 'add_product');
    formData.append('category_id', categoryId || '');
    formData.append('name', name);
    formData.append('description', description);
    formData.append('type', type);
    formData.append('price', price);
    
    fetch('add_product_ajax.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Ürün başarıyla kaydedildi, şimdi teklife ekle
            itemCounter++;
            const total = quantity * price;
            
            const item = {
                id: itemCounter,
                product_id: data.product_id, // Yeni eklenen ürünün ID'si
                name: name,
                description: description || name,
                quantity: quantity,
                price: price,
                line_total: total,
                category_name: data.category_name || 'Kategori Yok'
            };
            
            selectedItems.push(item);
            updateItemsTable();
            updateQuoteSummary();
            
            // Kategori bazlı ürün listesini güncelle
            if (categoryId) {
                loadCategoryProducts(categoryId);
            }
            
            // Formu temizle
            clearManualForm();
            
            // Başarı mesajı
            showSuccessMessage('Ürün başarıyla eklendi ve teklife dahil edildi!');
        } else {
            alert('Ürün eklenirken hata oluştu: ' + data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Bir hata oluştu. Lütfen tekrar deneyin.');
    });
}

// Manuel form temizleme
function clearManualForm() {
    document.getElementById('manual_category').value = '';
    document.getElementById('manual_name').value = '';
    document.getElementById('manual_quantity').value = '1';
    document.getElementById('manual_price').value = '';
    document.getElementById('manual_description').value = '';
    document.getElementById('manual_type').value = 'service';
}

// Başarı mesajı gösterme
function showSuccessMessage(message) {
    // Basit alert yerine daha güzel bir bildirim sistemi
    const alertDiv = document.createElement('div');
    alertDiv.className = 'alert alert-success alert-dismissible fade show position-fixed';
    alertDiv.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
    alertDiv.innerHTML = `
        <i class="fas fa-check-circle me-2"></i>${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    document.body.appendChild(alertDiv);
    
    // 3 saniye sonra otomatik kaldır
    setTimeout(() => {
        if (alertDiv.parentNode) {
            alertDiv.remove();
        }
    }, 3000);
}

// Başarı modalı göster
function showSuccessModal(data) {
    const modal = document.createElement('div');
    modal.className = 'modal fade';
    modal.innerHTML = `
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header bg-success text-white">
                    <h5 class="modal-title">
                        <i class="fas fa-check-circle me-2"></i>Teklif Başarıyla Oluşturuldu!
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="row">
                        <div class="col-md-6">
                            <h6 class="text-primary">Teklif Bilgileri</h6>
                            <ul class="list-unstyled">
                                <li><strong>Teklif No:</strong> ${data.quote_number}</li>
                                <li><strong>Teklif ID:</strong> ${data.quote_id}</li>
                                <li><strong>PDF Oluşturuldu:</strong> ${data.pdf_generated ? 'Evet' : 'Hayır'}</li>
                                <li><strong>E-posta Gönderildi:</strong> ${data.notification_sent ? 'Evet' : 'Hayır'}</li>
                            </ul>
                        </div>
                        <div class="col-md-6">
                            <h6 class="text-primary">Müşteri Linki</h6>
                            <div class="input-group mb-3">
                                <input type="text" class="form-control" value="${data.quote_link}" readonly id="quoteUrlInput">
                                <button class="btn btn-outline-primary" type="button" onclick="copyQuoteUrl()">
                                    <i class="fas fa-copy"></i> Kopyala
                                </button>
                            </div>
                            <small class="text-muted">Bu linki müşteriye göndererek teklifi görüntüleyebilir.</small>
                        </div>
                    </div>
                    
                    <div class="alert alert-info mt-3">
                        <i class="fas fa-info-circle me-2"></i>
                        Teklif başarıyla oluşturuldu ve müşteriye e-posta ile gönderildi. 
                        Yukarıdaki linki manuel olarak da paylaşabilirsiniz.
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Kapat</button>
                    <button type="button" class="btn btn-primary" onclick="goToQuotes()">
                        <i class="fas fa-list"></i> Teklif Listesine Git
                    </button>
                    <button type="button" class="btn btn-success" onclick="createNewQuote()">
                        <i class="fas fa-plus"></i> Yeni Teklif Oluştur
                    </button>
                </div>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
    const bsModal = new bootstrap.Modal(modal);
    bsModal.show();
    
    // Modal kapandığında temizle
    modal.addEventListener('hidden.bs.modal', () => {
        document.body.removeChild(modal);
    });
}

// Teklif URL'sini kopyala
function copyQuoteUrl() {
    const input = document.getElementById('quoteUrlInput');
    input.select();
    input.setSelectionRange(0, 99999);
    
    try {
        document.execCommand('copy');
        showSuccessMessage('Teklif linki panoya kopyalandı!');
    } catch (err) {
        console.error('Copy error:', err);
        alert('Kopyalama başarısız. Lütfen manuel olarak kopyalayın.');
    }
}

// Teklif listesine git
function goToQuotes() {
    window.location.href = 'quotes.php';
}

// Yeni teklif oluştur
function createNewQuote() {
    window.location.reload();
}

// Mevcut ürünü teklife ekle
function addProductToQuote(productId, name, price, description) {
    addItemToQuote(productId, name, description, 1, price);
}

// Ürünü teklife ekle
function addItemToQuote(productId, name, description, quantity, price) {
    itemCounter++;
    const total = quantity * price;
    
    const item = {
        id: itemCounter,
        product_id: productId,
        name: name,
        description: description,
        quantity: quantity,
        price: price, // Hem unit_price hem de price olarak kullanılacak
        line_total: total
    };
    
    selectedItems.push(item);
    updateItemsTable();
    updateQuoteSummary();
}

// Ürünü kaldır
function removeItem(itemId) {
    selectedItems = selectedItems.filter(item => item.id !== itemId);
    updateItemsTable();
    updateQuoteSummary();
}

// Adet güncelle
function updateQuantity(itemId, newQuantity) {
    const item = selectedItems.find(item => item.id === itemId);
    if (item && newQuantity > 0) {
        item.quantity = newQuantity;
        item.line_total = item.quantity * item.price;
        updateItemsTable();
        updateQuoteSummary();
    }
}

// Ürünler tablosunu güncelle
function updateItemsTable() {
    const tbody = document.getElementById('items_tbody');
    const noItemsRow = document.getElementById('no_items_row');
    
    if (selectedItems.length === 0) {
        noItemsRow.style.display = 'table-row';
        return;
    }
    
    noItemsRow.style.display = 'none';
    
    let html = '';
    selectedItems.forEach(item => {
        html += `
            <tr>
                <td>
                    <strong>${escapeHtml(item.name)}</strong>
                    <input type="hidden" name="items[${item.id}][product_id]" value="${item.product_id || ''}">
                    <input type="hidden" name="items[${item.id}][name]" value="${escapeHtml(item.name)}">
                </td>
                <td>
                    <small class="text-muted">${escapeHtml(item.description || '')}</small>
                    <input type="hidden" name="items[${item.id}][description]" value="${escapeHtml(item.description || '')}">
                </td>
                <td>
                    <input type="number" class="form-control form-control-sm" style="width: 80px;" 
                           value="${item.quantity}" min="1" 
                           onchange="updateQuantity(${item.id}, this.value)"
                           name="items[${item.id}][quantity]">
                </td>
                <td>
                    ₺${item.price.toFixed(2)}
                    <input type="hidden" name="items[${item.id}][unit_price]" value="${item.price}">
                </td>
                <td>
                    <strong>₺${item.line_total.toFixed(2)}</strong>
                    <input type="hidden" name="items[${item.id}][line_total]" value="${item.line_total}">
                </td>
                <td>
                    <button type="button" class="btn btn-sm btn-outline-danger" onclick="removeItem(${item.id})">
                        <i class="fas fa-trash"></i>
                    </button>
                </td>
            </tr>
        `;
    });
    
    tbody.innerHTML = html + noItemsRow.outerHTML;
}

// Teklif özetini güncelle
function updateQuoteSummary() {
    const subtotal = selectedItems.reduce((sum, item) => sum + item.line_total, 0);
    const discountRate = parseFloat(document.getElementById('discount_rate').value) || 0;
    const taxRate = parseFloat(document.getElementById('tax_rate').value) || 0;
    
    const discountAmount = subtotal * (discountRate / 100);
    const taxableAmount = subtotal - discountAmount;
    const taxAmount = taxableAmount * (taxRate / 100);
    const total = taxableAmount + taxAmount;
    
    document.getElementById('subtotal').textContent = `₺${subtotal.toFixed(2)}`;
    document.getElementById('discount_amount').textContent = `₺${discountAmount.toFixed(2)}`;
    document.getElementById('tax_amount').textContent = `₺${taxAmount.toFixed(2)}`;
    document.getElementById('total_amount').textContent = `₺${total.toFixed(2)}`;
}

// Ürün filtreleme
function filterProducts() {
    const categoryFilter = document.getElementById('category_filter').value;
    const searchFilter = document.getElementById('product_search').value.toLowerCase();
    const products = document.querySelectorAll('.product-item');
    
    products.forEach(product => {
        const category = product.dataset.category;
        const name = product.dataset.name;
        
        const categoryMatch = !categoryFilter || category === categoryFilter;
        const nameMatch = !searchFilter || name.includes(searchFilter);
        
        product.style.display = categoryMatch && nameMatch ? 'block' : 'none';
    });
}

// HTML escape
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Kategori değiştiğinde ürünleri yükle
function loadCategoryProducts(categoryId) {
    if (!categoryId) {
        document.getElementById('categoryProducts').innerHTML = '<p class="text-muted">Kategori seçin</p>';
        return;
    }
    
    // Loading göster
    document.getElementById('categoryProducts').innerHTML = '<div class="text-center"><i class="fas fa-spinner fa-spin"></i> Yükleniyor...</div>';
    
    fetch(`get_products.php?category_id=${categoryId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                let html = '';
                if (data.products.length === 0) {
                    html = '<p class="text-muted">Bu kategoride ürün bulunamadı</p>';
                } else {
                    html = '<div class="row">';
                    data.products.forEach(product => {
                        html += `
                            <div class="col-md-6 mb-2">
                                <div class="card h-100 product-card" style="cursor: pointer; transition: all 0.3s ease;" 
                                     onclick="addProductToQuote(${product.id}, '${escapeHtml(product.name)}', ${product.price}, '${escapeHtml(product.description || '')}')"
                                     onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 8px rgba(0,0,0,0.1)'"
                                     onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='0 1px 3px rgba(0,0,0,0.1)'">
                                    <div class="card-body p-3">
                                        <h6 class="card-title mb-1 text-primary">${escapeHtml(product.name)}</h6>
                                        <p class="card-text small text-muted mb-2">${escapeHtml(product.description || 'Açıklama yok')}</p>
                                        <div class="d-flex justify-content-between align-items-center">
                                            <span class="badge ${getTypeBadgeClass(product.type)}">${getTypeLabel(product.type)}</span>
                                            <strong class="text-success">₺${parseFloat(product.price).toFixed(2)}</strong>
                                        </div>
                                        <div class="mt-2">
                                            <small class="text-muted">
                                                <i class="fas fa-plus-circle me-1"></i>Teklife eklemek için tıklayın
                                            </small>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        `;
                    });
                    html += '</div>';
                }
                document.getElementById('categoryProducts').innerHTML = html;
            } else {
                document.getElementById('categoryProducts').innerHTML = '<div class="alert alert-warning"><i class="fas fa-exclamation-triangle me-2"></i>Ürünler yüklenirken hata oluştu</div>';
            }
        })
        .catch(error => {
            console.error('Error:', error);
            document.getElementById('categoryProducts').innerHTML = '<div class="alert alert-danger"><i class="fas fa-times-circle me-2"></i>Ürünler yüklenirken hata oluştu</div>';
        });
}

// Ürün türü için badge sınıfı
function getTypeBadgeClass(type) {
    switch(type) {
        case 'service': return 'bg-primary';
        case 'part': return 'bg-success';
        case 'package': return 'bg-warning text-dark';
        default: return 'bg-secondary';
    }
}

// Ürün türü için etiket
function getTypeLabel(type) {
    switch(type) {
        case 'service': return 'Hizmet';
        case 'part': return 'Parça';
        case 'package': return 'Paket';
        default: return 'Diğer';
    }
}

// Form gönderimi
document.getElementById('quoteForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    // Seçili ürünleri form'a ekle
    const formData = new FormData(this);
    
    // Seçili ürünleri ekle
    selectedItems.forEach((item, index) => {
        if (item.product_id) {
            // Mevcut ürün
            formData.append(`products[${index}][id]`, item.product_id);
            formData.append(`products[${index}][name]`, item.name);
            formData.append(`products[${index}][quantity]`, item.quantity);
            formData.append(`products[${index}][price]`, item.price);
        } else {
            // Manuel ürün
            formData.append(`manual_items[${index}][name]`, item.name);
            formData.append(`manual_items[${index}][description]`, item.description || item.name);
            formData.append(`manual_items[${index}][quantity]`, item.quantity);
            formData.append(`manual_items[${index}][unit_price]`, item.price);
        }
    });
    
    const submitBtn = document.getElementById('submitBtn');
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Gönderiliyor...';
    
    // AJAX ile gönder
    fetch('process_create_quote.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Başarı modalı göster
            showSuccessModal(data);
        } else {
            alert('Hata: ' + data.message);
            submitBtn.disabled = false;
            submitBtn.innerHTML = '<i class="fas fa-paper-plane"></i> Teklif Oluştur';
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Bir hata oluştu. Lütfen tekrar deneyin.');
        submitBtn.disabled = false;
        submitBtn.innerHTML = '<i class="fas fa-paper-plane"></i> Teklif Oluştur';
    });
});

// Sayfa yüklendiğinde
document.addEventListener('DOMContentLoaded', function() {
    // İndirim ve KDV oranı değiştiğinde özeti güncelle
    document.getElementById('discount_rate').addEventListener('input', updateQuoteSummary);
    document.getElementById('tax_rate').addEventListener('input', updateQuoteSummary);
    
    // İlk özeti göster
    updateQuoteSummary();
});
</script>

<?php require_once 'includes/footer.php'; ?>