<?php
session_start();
require_once '../config/config.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');

function jsonResponse($data, $status = 200) {
    http_response_code($status);
    echo json_encode($data);
    exit;
}

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'message' => 'Oturum geçersiz.'], 401);
}

$action = $_POST['action'] ?? $_GET['action'] ?? '';

try {
    switch ($action) {
        case 'add':
            $name = trim($_POST['name'] ?? '');
            $description = trim($_POST['description'] ?? '');
            $sort_order = intval($_POST['sort_order'] ?? 0);
            $is_active = intval($_POST['is_active'] ?? 1);
            
            if (empty($name)) {
                jsonResponse(['success' => false, 'message' => 'Kategori adı gereklidir.']);
            }
            
            // Aynı isimde kategori var mı kontrol et
            $stmt = $pdo->prepare("SELECT id FROM product_categories WHERE name = ?");
            $stmt->execute([$name]);
            if ($stmt->fetch()) {
                jsonResponse(['success' => false, 'message' => 'Bu isimde bir kategori zaten mevcut.']);
            }
            
            // Kategori ekle
            $stmt = $pdo->prepare("
                INSERT INTO product_categories (name, description, sort_order, is_active, created_at) 
                VALUES (?, ?, ?, ?, NOW())
            ");
            $stmt->execute([$name, $description, $sort_order, $is_active]);
            
            $category_id = $pdo->lastInsertId();
            
            // Audit log
            logAudit('category_created', 'product_categories', $category_id, null, [
                'name' => $name,
                'description' => $description,
                'sort_order' => $sort_order,
                'is_active' => $is_active,
                'created_by' => $_SESSION['user_id']
            ]);
            
            jsonResponse([
                'success' => true, 
                'message' => 'Kategori başarıyla eklendi.',
                'category_id' => $category_id
            ]);
            break;
            
        case 'get':
            $id = intval($_GET['id'] ?? 0);
            
            if ($id <= 0) {
                jsonResponse(['success' => false, 'message' => 'Geçersiz kategori ID.']);
            }
            
            $stmt = $pdo->prepare("SELECT * FROM product_categories WHERE id = ?");
            $stmt->execute([$id]);
            $category = $stmt->fetch();
            
            if (!$category) {
                jsonResponse(['success' => false, 'message' => 'Kategori bulunamadı.']);
            }
            
            jsonResponse([
                'success' => true,
                'category' => $category
            ]);
            break;
            
        case 'update':
            $id = intval($_POST['id'] ?? 0);
            $name = trim($_POST['name'] ?? '');
            $description = trim($_POST['description'] ?? '');
            $sort_order = intval($_POST['sort_order'] ?? 0);
            $is_active = intval($_POST['is_active'] ?? 1);
            
            if ($id <= 0) {
                jsonResponse(['success' => false, 'message' => 'Geçersiz kategori ID.']);
            }
            
            if (empty($name)) {
                jsonResponse(['success' => false, 'message' => 'Kategori adı gereklidir.']);
            }
            
            // Mevcut kategori bilgilerini al
            $stmt = $pdo->prepare("SELECT * FROM product_categories WHERE id = ?");
            $stmt->execute([$id]);
            $old_category = $stmt->fetch();
            
            if (!$old_category) {
                jsonResponse(['success' => false, 'message' => 'Kategori bulunamadı.']);
            }
            
            // Aynı isimde başka kategori var mı kontrol et
            $stmt = $pdo->prepare("SELECT id FROM product_categories WHERE name = ? AND id != ?");
            $stmt->execute([$name, $id]);
            if ($stmt->fetch()) {
                jsonResponse(['success' => false, 'message' => 'Bu isimde bir kategori zaten mevcut.']);
            }
            
            // Kategoriyi güncelle
            $stmt = $pdo->prepare("
                UPDATE product_categories 
                SET name = ?, description = ?, sort_order = ?, is_active = ?, updated_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$name, $description, $sort_order, $is_active, $id]);
            
            // Audit log
            logAudit('category_updated', 'product_categories', $id, $old_category, [
                'name' => $name,
                'description' => $description,
                'sort_order' => $sort_order,
                'is_active' => $is_active,
                'updated_by' => $_SESSION['user_id']
            ]);
            
            jsonResponse([
                'success' => true, 
                'message' => 'Kategori başarıyla güncellendi.'
            ]);
            break;
            
        case 'toggle_status':
            $id = intval($_POST['id'] ?? 0);
            $status = intval($_POST['status'] ?? 0);
            
            if ($id <= 0) {
                jsonResponse(['success' => false, 'message' => 'Geçersiz kategori ID.']);
            }
            
            // Mevcut kategori bilgilerini al
            $stmt = $pdo->prepare("SELECT * FROM product_categories WHERE id = ?");
            $stmt->execute([$id]);
            $old_category = $stmt->fetch();
            
            if (!$old_category) {
                jsonResponse(['success' => false, 'message' => 'Kategori bulunamadı.']);
            }
            
            // Durumu güncelle
            $stmt = $pdo->prepare("UPDATE product_categories SET is_active = ?, updated_at = NOW() WHERE id = ?");
            $stmt->execute([$status, $id]);
            
            // Audit log
            logAudit('category_status_changed', 'product_categories', $id, $old_category, [
                'old_status' => $old_category['is_active'],
                'new_status' => $status,
                'updated_by' => $_SESSION['user_id']
            ]);
            
            $status_text = $status ? 'aktif' : 'pasif';
            jsonResponse([
                'success' => true, 
                'message' => "Kategori başarıyla {$status_text} yapıldı."
            ]);
            break;
            
        case 'delete':
            $id = intval($_POST['id'] ?? 0);
            
            if ($id <= 0) {
                jsonResponse(['success' => false, 'message' => 'Geçersiz kategori ID.']);
            }
            
            // Mevcut kategori bilgilerini al
            $stmt = $pdo->prepare("SELECT * FROM product_categories WHERE id = ?");
            $stmt->execute([$id]);
            $category = $stmt->fetch();
            
            if (!$category) {
                jsonResponse(['success' => false, 'message' => 'Kategori bulunamadı.']);
            }
            
            // Bu kategoriye ait ürün var mı kontrol et
            $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM products WHERE category_id = ?");
            $stmt->execute([$id]);
            $product_count = $stmt->fetch()['count'];
            
            if ($product_count > 0) {
                jsonResponse(['success' => false, 'message' => 'Bu kategoriye ait ürünler bulunduğu için silinemez.']);
            }
            
            // Kategoriyi sil
            $stmt = $pdo->prepare("DELETE FROM product_categories WHERE id = ?");
            $stmt->execute([$id]);
            
            // Audit log
            logAudit('category_deleted', 'product_categories', $id, $category, [
                'deleted_by' => $_SESSION['user_id']
            ]);
            
            jsonResponse([
                'success' => true, 
                'message' => 'Kategori başarıyla silindi.'
            ]);
            break;
            
        default:
            jsonResponse(['success' => false, 'message' => 'Geçersiz işlem.'], 400);
    }
    
} catch (Exception $e) {
    error_log("Category action error: " . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Bir hata oluştu: ' . $e->getMessage()], 500);
}
?>
