<?php
session_start();
require_once '../config/config.php';
require_once '../includes/functions.php';
require_once '../includes/appointment_functions.php';

$pageTitle = 'Randevular';
$currentPage = 'appointments';

if (!isLoggedIn()) {
    redirect('login.php');
}

// Filtreleme
$filters = [];
if (!empty($_GET['date_from'])) {
    $filters['date_from'] = $_GET['date_from'];
}
if (!empty($_GET['date_to'])) {
    $filters['date_to'] = $_GET['date_to'];
}
if (!empty($_GET['status'])) {
    $filters['status'] = $_GET['status'];
}
if (!empty($_GET['search'])) {
    $filters['search'] = $_GET['search'];
}

// Sayfalama
$page = max(1, intval($_GET['page'] ?? 1));
$limit = 20;
$offset = ($page - 1) * $limit;

// Randevuları getir
$appointments = getAppointments($filters, $limit, $offset);
$totalAppointments = getAppointmentsCount($filters);
$totalPages = ceil($totalAppointments / $limit);

// İstatistikler
$todayCount = getDailyAppointmentCount(date('Y-m-d'));
$tomorrowCount = getDailyAppointmentCount(date('Y-m-d', strtotime('+1 day')));

$stmt = $pdo->prepare("SELECT COUNT(*) FROM appointments WHERE status = 'PENDING'");
$stmt->execute();
$pendingCount = $stmt->fetchColumn();

$stmt = $pdo->prepare("SELECT COUNT(*) FROM appointments WHERE appointment_date = CURDATE() AND status = 'CONFIRMED'");
$stmt->execute();
$todayConfirmedCount = $stmt->fetchColumn();

include 'includes/header.php';
?>

<div class="container-fluid">
    <!-- İstatistik Kartları -->
    <div class="row mb-4">
        <div class="col-md-3">
            <div class="nice-card bg-primary text-white">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h6 class="card-title mb-1">Bugün</h6>
                            <h3 class="mb-0"><?= $todayCount ?></h3>
                            <small>Toplam Randevu</small>
                        </div>
                        <div class="card-icon">
                            <i class="fas fa-calendar-day fa-2x opacity-75"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="col-md-3">
            <div class="nice-card bg-info text-white">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h6 class="card-title mb-1">Yarın</h6>
                            <h3 class="mb-0"><?= $tomorrowCount ?></h3>
                            <small>Toplam Randevu</small>
                        </div>
                        <div class="card-icon">
                            <i class="fas fa-calendar-plus fa-2x opacity-75"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="col-md-3">
            <div class="nice-card bg-warning text-white">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h6 class="card-title mb-1">Bekleyen</h6>
                            <h3 class="mb-0"><?= $pendingCount ?></h3>
                            <small>Onay Bekliyor</small>
                        </div>
                        <div class="card-icon">
                            <i class="fas fa-clock fa-2x opacity-75"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="col-md-3">
            <div class="nice-card bg-success text-white">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h6 class="card-title mb-1">Bugün Onaylı</h6>
                            <h3 class="mb-0"><?= $todayConfirmedCount ?></h3>
                            <small>Kesin Randevu</small>
                        </div>
                        <div class="card-icon">
                            <i class="fas fa-check-circle fa-2x opacity-75"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Randevular Tablosu -->
    <div class="nice-card">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h5 class="card-title mb-0">
                <i class="fas fa-calendar-alt text-primary me-2"></i>
                Randevular
            </h5>
            <div class="d-flex gap-2">
                <button class="nice-btn nice-btn-outline-primary" data-bs-toggle="modal" data-bs-target="#filterModal">
                    <i class="fas fa-filter me-2"></i>Filtrele
                </button>
                <a href="appointment_add.php" class="nice-btn nice-btn-primary">
                    <i class="fas fa-plus me-2"></i>Yeni Randevu
                </a>
            </div>
        </div>
        
        <div class="card-body">
            <?php if (empty($appointments)): ?>
            <div class="text-center py-5">
                <i class="fas fa-calendar-times fa-3x text-muted mb-3"></i>
                <h5 class="text-muted">Randevu bulunamadı</h5>
                <p class="text-muted">Henüz hiç randevu kaydı yok veya arama kriterlerinize uygun randevu bulunamadı.</p>
            </div>
            <?php else: ?>
            <div class="table-responsive">
                <table class="nice-table">
                    <thead>
                        <tr>
                            <th>Tarih & Saat</th>
                            <th>Müşteri</th>
                            <th>Araç</th>
                            <th>Hizmet</th>
                            <th>Durum</th>
                            <th>Teknisyen</th>
                            <th>İşlemler</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($appointments as $appointment): ?>
                        <tr>
                            <td>
                                <div class="fw-bold"><?= formatDate($appointment['appointment_date'], 'd.m.Y') ?></div>
                                <small class="text-muted"><?= date('H:i', strtotime($appointment['appointment_time'])) ?></small>
                            </td>
                            <td>
                                <div class="fw-bold"><?= h($appointment['first_name'] . ' ' . $appointment['last_name']) ?></div>
                                <small class="text-muted">
                                    <i class="fas fa-phone me-1"></i><?= h($appointment['phone']) ?>
                                </small>
                            </td>
                            <td>
                                <div class="fw-bold"><?= h($appointment['brand'] . ' ' . $appointment['model']) ?></div>
                                <small class="text-muted"><?= h($appointment['year'] . ' - ' . $appointment['plate']) ?></small>
                            </td>
                            <td>
                                <span class="nice-badge nice-badge-info"><?= h($appointment['service_type']) ?></span>
                            </td>
                            <td>
                                <span class="nice-badge nice-badge-<?= getAppointmentStatusClass($appointment['status']) ?>">
                                    <?= getAppointmentStatusText($appointment['status']) ?>
                                </span>
                            </td>
                            <td>
                                <?php if ($appointment['tech_first_name']): ?>
                                    <span class="text-success">
                                        <i class="fas fa-user-check me-1"></i>
                                        <?= h($appointment['tech_first_name'] . ' ' . $appointment['tech_last_name']) ?>
                                    </span>
                                <?php else: ?>
                                    <span class="text-muted">
                                        <i class="fas fa-user-times me-1"></i>Atanmadı
                                    </span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <div class="dropdown">
                                    <button class="nice-btn nice-btn-sm nice-btn-outline-secondary dropdown-toggle" 
                                            type="button" data-bs-toggle="dropdown">
                                        İşlemler
                                    </button>
                                    <ul class="dropdown-menu">
                                        <li>
                                            <a class="dropdown-item" href="appointment_detail.php?id=<?= $appointment['id'] ?>">
                                                <i class="fas fa-eye me-2"></i>Detay
                                            </a>
                                        </li>
                                        <?php if ($appointment['status'] === 'PENDING'): ?>
                                        <li>
                                            <a class="dropdown-item text-success" href="#" 
                                               onclick="updateStatus(<?= $appointment['id'] ?>, 'CONFIRMED')">
                                                <i class="fas fa-check me-2"></i>Onayla
                                            </a>
                                        </li>
                                        <?php endif; ?>
                                        <?php if (in_array($appointment['status'], ['PENDING', 'CONFIRMED'])): ?>
                                        <li>
                                            <a class="dropdown-item text-primary" href="#" 
                                               onclick="updateStatus(<?= $appointment['id'] ?>, 'IN_PROGRESS')">
                                                <i class="fas fa-play me-2"></i>Başlat
                                            </a>
                                        </li>
                                        <?php endif; ?>
                                        <?php if ($appointment['status'] === 'IN_PROGRESS'): ?>
                                        <li>
                                            <a class="dropdown-item text-success" href="#" 
                                               onclick="updateStatus(<?= $appointment['id'] ?>, 'COMPLETED')">
                                                <i class="fas fa-check-circle me-2"></i>Tamamla
                                            </a>
                                        </li>
                                        <?php endif; ?>
                                        <li><hr class="dropdown-divider"></li>
                                        <?php if (!$appointment['reminder_sent'] && $appointment['status'] !== 'CANCELLED'): ?>
                                        <li>
                                            <a class="dropdown-item text-info" href="#" 
                                               onclick="sendReminder(<?= $appointment['id'] ?>)">
                                                <i class="fas fa-bell me-2"></i>Hatırlatma Gönder
                                            </a>
                                        </li>
                                        <?php endif; ?>
                                        <li>
                                            <a class="dropdown-item text-danger" href="#" 
                                               onclick="updateStatus(<?= $appointment['id'] ?>, 'CANCELLED')">
                                                <i class="fas fa-times me-2"></i>İptal Et
                                            </a>
                                        </li>
                                    </ul>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            
            <!-- Sayfalama -->
            <?php if ($totalPages > 1): ?>
            <nav class="mt-4">
                <ul class="pagination justify-content-center">
                    <?php if ($page > 1): ?>
                    <li class="page-item">
                        <a class="page-link" href="?page=<?= $page - 1 ?><?= http_build_query($filters) ? '&' . http_build_query($filters) : '' ?>">Önceki</a>
                    </li>
                    <?php endif; ?>
                    
                    <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                    <li class="page-item <?= $i === $page ? 'active' : '' ?>">
                        <a class="page-link" href="?page=<?= $i ?><?= http_build_query($filters) ? '&' . http_build_query($filters) : '' ?>"><?= $i ?></a>
                    </li>
                    <?php endfor; ?>
                    
                    <?php if ($page < $totalPages): ?>
                    <li class="page-item">
                        <a class="page-link" href="?page=<?= $page + 1 ?><?= http_build_query($filters) ? '&' . http_build_query($filters) : '' ?>">Sonraki</a>
                    </li>
                    <?php endif; ?>
                </ul>
            </nav>
            <?php endif; ?>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- Filtre Modal -->
<div class="modal fade" id="filterModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Randevu Filtrele</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="GET">
                <div class="modal-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Başlangıç Tarihi</label>
                                <input type="date" class="nice-form-control" name="date_from" 
                                       value="<?= h($_GET['date_from'] ?? '') ?>">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Bitiş Tarihi</label>
                                <input type="date" class="nice-form-control" name="date_to" 
                                       value="<?= h($_GET['date_to'] ?? '') ?>">
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Durum</label>
                        <select class="nice-form-control" name="status">
                            <option value="">Tüm Durumlar</option>
                            <option value="PENDING" <?= ($_GET['status'] ?? '') === 'PENDING' ? 'selected' : '' ?>>Beklemede</option>
                            <option value="CONFIRMED" <?= ($_GET['status'] ?? '') === 'CONFIRMED' ? 'selected' : '' ?>>Onaylandı</option>
                            <option value="IN_PROGRESS" <?= ($_GET['status'] ?? '') === 'IN_PROGRESS' ? 'selected' : '' ?>>Devam Ediyor</option>
                            <option value="COMPLETED" <?= ($_GET['status'] ?? '') === 'COMPLETED' ? 'selected' : '' ?>>Tamamlandı</option>
                            <option value="CANCELLED" <?= ($_GET['status'] ?? '') === 'CANCELLED' ? 'selected' : '' ?>>İptal Edildi</option>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Arama</label>
                        <input type="text" class="nice-form-control" name="search" 
                               placeholder="Müşteri adı, plaka..." value="<?= h($_GET['search'] ?? '') ?>">
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="nice-btn nice-btn-secondary" data-bs-dismiss="modal">İptal</button>
                    <button type="submit" class="nice-btn nice-btn-primary">Filtrele</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
// Randevu durumu güncelle
function updateStatus(appointmentId, status) {
    if (confirm('Randevu durumunu değiştirmek istediğinizden emin misiniz?')) {
        fetch('appointment_actions.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                action: 'update_status',
                appointment_id: appointmentId,
                status: status
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                location.reload();
            } else {
                alert('Hata: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Bir hata oluştu.');
        });
    }
}

// Hatırlatma gönder
function sendReminder(appointmentId) {
    if (confirm('Müşteriye randevu hatırlatması göndermek istediğinizden emin misiniz?')) {
        fetch('appointment_actions.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                action: 'send_reminder',
                appointment_id: appointmentId
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Hatırlatma başarıyla gönderildi.');
                location.reload();
            } else {
                alert('Hata: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Bir hata oluştu.');
        });
    }
}
</script>

<?php include 'includes/footer.php'; ?>
