<?php
session_start();
require_once '../config/config.php';
require_once '../includes/functions.php';
require_once '../includes/appointment_functions.php';

$pageTitle = 'Yeni Randevu';
$currentPage = 'appointments';

if (!isLoggedIn()) {
    redirect('login.php');
}

$message = '';
$messageType = '';

// Form gönderildiğinde
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $customer_id = intval($_POST['customer_id'] ?? 0);
        $vehicle_id = intval($_POST['vehicle_id'] ?? 0);
        $appointment_date = $_POST['appointment_date'] ?? '';
        $appointment_time = $_POST['appointment_time'] ?? '';
        $service_type = trim($_POST['service_type'] ?? '');
        $duration_minutes = intval($_POST['duration_minutes'] ?? 60);
        $notes = trim($_POST['notes'] ?? '');
        $assigned_technician_id = intval($_POST['assigned_technician_id'] ?? 0) ?: null;
        $status = $_POST['status'] ?? 'PENDING';
        
        // Validasyon
        if (!$customer_id) {
            throw new Exception('Lütfen müşteri seçin.');
        }
        
        if (!$vehicle_id) {
            throw new Exception('Lütfen araç seçin.');
        }
        
        if (empty($appointment_date)) {
            throw new Exception('Lütfen randevu tarihi seçin.');
        }
        
        if (empty($appointment_time)) {
            throw new Exception('Lütfen randevu saati seçin.');
        }
        
        if (empty($service_type)) {
            throw new Exception('Lütfen hizmet türü girin.');
        }
        
        // Tarih kontrolü
        if (strtotime($appointment_date) < strtotime('today')) {
            throw new Exception('Geçmiş tarih seçilemez.');
        }
        
        // Randevu verilerini hazırla
        $appointmentData = [
            'customer_id' => $customer_id,
            'vehicle_id' => $vehicle_id,
            'appointment_date' => $appointment_date,
            'appointment_time' => $appointment_time,
            'duration_minutes' => $duration_minutes,
            'service_type' => $service_type,
            'notes' => $notes
        ];
        
        // Randevuyu oluştur
        $appointmentId = createAppointment($appointmentData);
        
        // Teknisyen ataması varsa
        if ($assigned_technician_id) {
            $stmt = $pdo->prepare("UPDATE appointments SET assigned_technician_id = ? WHERE id = ?");
            $stmt->execute([$assigned_technician_id, $appointmentId]);
        }
        
        // Durum güncelle (PENDING dışında bir durum seçildiyse)
        if ($status !== 'PENDING') {
            updateAppointmentStatus($appointmentId, $status);
        }
        
        // Audit log
        logAudit('appointment_created', 'appointments', $appointmentId, $_SESSION['user_id'], [
            'customer_id' => $customer_id,
            'vehicle_id' => $vehicle_id,
            'appointment_date' => $appointment_date,
            'appointment_time' => $appointment_time,
            'service_type' => $service_type,
            'created_by' => $_SESSION['user_id']
        ]);
        
        $message = 'Randevu başarıyla oluşturuldu.';
        $messageType = 'success';
        
        // 2 saniye sonra randevular sayfasına yönlendir
        header("refresh:2;url=appointments.php");
        
    } catch (Exception $e) {
        $message = 'Hata: ' . $e->getMessage();
        $messageType = 'danger';
        error_log("Appointment creation error: " . $e->getMessage());
    }
}

// Müşterileri getir
$stmt = $pdo->prepare("SELECT id, first_name, last_name, phone FROM customers ORDER BY first_name, last_name");
$stmt->execute();
$customers = $stmt->fetchAll();

// Teknisyenleri getir
$stmt = $pdo->prepare("SELECT id, first_name, last_name FROM users WHERE role IN ('admin', 'technician') ORDER BY first_name, last_name");
$stmt->execute();
$technicians = $stmt->fetchAll();

include 'includes/header.php';
?>

<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="nice-card">
                <div class="card-header">
                    <h5 class="card-title mb-0">
                        <i class="fas fa-calendar-plus text-primary me-2"></i>
                        Yeni Randevu Oluştur
                    </h5>
                </div>
                
                <div class="card-body">
                    <?php if ($message): ?>
                    <div class="alert alert-<?= $messageType ?> alert-dismissible fade show" role="alert">
                        <?= htmlspecialchars($message) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                    <?php endif; ?>
                    
                    <form method="POST" class="needs-validation" novalidate>
                        <div class="row">
                            <!-- Müşteri Seçimi -->
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label">Müşteri *</label>
                                    <select class="nice-form-control" name="customer_id" id="customer_id" required onchange="loadCustomerVehicles()">
                                        <option value="">Müşteri seçin</option>
                                        <?php foreach ($customers as $customer): ?>
                                        <option value="<?= $customer['id'] ?>" <?= (($_POST['customer_id'] ?? '') == $customer['id']) ? 'selected' : '' ?>>
                                            <?= h($customer['first_name'] . ' ' . $customer['last_name']) ?> - <?= h($customer['phone']) ?>
                                        </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <div class="invalid-feedback">Lütfen müşteri seçin.</div>
                                </div>
                            </div>
                            
                            <!-- Araç Seçimi -->
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label">Araç *</label>
                                    <select class="nice-form-control" name="vehicle_id" id="vehicle_id" required>
                                        <option value="">Önce müşteri seçin</option>
                                    </select>
                                    <div class="invalid-feedback">Lütfen araç seçin.</div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <!-- Randevu Tarihi -->
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label class="form-label">Randevu Tarihi *</label>
                                    <input type="date" class="nice-form-control" name="appointment_date" id="appointment_date" 
                                           value="<?= h($_POST['appointment_date'] ?? '') ?>" required onchange="loadAvailableSlots()">
                                    <div class="invalid-feedback">Lütfen randevu tarihi seçin.</div>
                                </div>
                            </div>
                            
                            <!-- Randevu Saati -->
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label class="form-label">Randevu Saati *</label>
                                    <select class="nice-form-control" name="appointment_time" id="appointment_time" required>
                                        <option value="">Önce tarih seçin</option>
                                    </select>
                                    <div class="invalid-feedback">Lütfen randevu saati seçin.</div>
                                </div>
                            </div>
                            
                            <!-- Süre -->
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label class="form-label">Süre (Dakika)</label>
                                    <select class="nice-form-control" name="duration_minutes">
                                        <option value="30" <?= (($_POST['duration_minutes'] ?? 60) == 30) ? 'selected' : '' ?>>30 Dakika</option>
                                        <option value="60" <?= (($_POST['duration_minutes'] ?? 60) == 60) ? 'selected' : '' ?>>1 Saat</option>
                                        <option value="90" <?= (($_POST['duration_minutes'] ?? 60) == 90) ? 'selected' : '' ?>>1.5 Saat</option>
                                        <option value="120" <?= (($_POST['duration_minutes'] ?? 60) == 120) ? 'selected' : '' ?>>2 Saat</option>
                                        <option value="180" <?= (($_POST['duration_minutes'] ?? 60) == 180) ? 'selected' : '' ?>>3 Saat</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <!-- Hizmet Türü -->
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label">Hizmet Türü *</label>
                                    <select class="nice-form-control" name="service_type" required>
                                        <option value="">Hizmet türü seçin</option>
                                        <option value="Periyodik Bakım" <?= (($_POST['service_type'] ?? '') == 'Periyodik Bakım') ? 'selected' : '' ?>>Periyodik Bakım</option>
                                        <option value="Motor Onarımı" <?= (($_POST['service_type'] ?? '') == 'Motor Onarımı') ? 'selected' : '' ?>>Motor Onarımı</option>
                                        <option value="Kaporta & Boya" <?= (($_POST['service_type'] ?? '') == 'Kaporta & Boya') ? 'selected' : '' ?>>Kaporta & Boya</option>
                                        <option value="Fren Sistemi" <?= (($_POST['service_type'] ?? '') == 'Fren Sistemi') ? 'selected' : '' ?>>Fren Sistemi</option>
                                        <option value="Klima Servisi" <?= (($_POST['service_type'] ?? '') == 'Klima Servisi') ? 'selected' : '' ?>>Klima Servisi</option>
                                        <option value="Lastik Değişimi" <?= (($_POST['service_type'] ?? '') == 'Lastik Değişimi') ? 'selected' : '' ?>>Lastik Değişimi</option>
                                        <option value="Genel Onarım" <?= (($_POST['service_type'] ?? '') == 'Genel Onarım') ? 'selected' : '' ?>>Genel Onarım</option>
                                        <option value="Diğer" <?= (($_POST['service_type'] ?? '') == 'Diğer') ? 'selected' : '' ?>>Diğer</option>
                                    </select>
                                    <div class="invalid-feedback">Lütfen hizmet türü seçin.</div>
                                </div>
                            </div>
                            
                            <!-- Teknisyen Ataması -->
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label">Teknisyen Ataması</label>
                                    <select class="nice-form-control" name="assigned_technician_id">
                                        <option value="">Teknisyen atanmadı</option>
                                        <?php foreach ($technicians as $technician): ?>
                                        <option value="<?= $technician['id'] ?>" <?= (($_POST['assigned_technician_id'] ?? '') == $technician['id']) ? 'selected' : '' ?>>
                                            <?= h($technician['first_name'] . ' ' . $technician['last_name']) ?>
                                        </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <!-- Durum -->
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label">Randevu Durumu</label>
                                    <select class="nice-form-control" name="status">
                                        <option value="PENDING" <?= (($_POST['status'] ?? 'PENDING') == 'PENDING') ? 'selected' : '' ?>>Beklemede</option>
                                        <option value="CONFIRMED" <?= (($_POST['status'] ?? '') == 'CONFIRMED') ? 'selected' : '' ?>>Onaylandı</option>
                                        <option value="IN_PROGRESS" <?= (($_POST['status'] ?? '') == 'IN_PROGRESS') ? 'selected' : '' ?>>Devam Ediyor</option>
                                        <option value="COMPLETED" <?= (($_POST['status'] ?? '') == 'COMPLETED') ? 'selected' : '' ?>>Tamamlandı</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Notlar -->
                        <div class="mb-3">
                            <label class="form-label">Randevu Notları</label>
                            <textarea class="nice-form-control" name="notes" rows="3" 
                                    placeholder="Randevu hakkında özel notlar..."><?= h($_POST['notes'] ?? '') ?></textarea>
                        </div>
                        
                        <!-- Butonlar -->
                        <div class="d-flex gap-2">
                            <button type="submit" class="nice-btn nice-btn-primary">
                                <i class="fas fa-save me-2"></i>Randevu Oluştur
                            </button>
                            <a href="appointments.php" class="nice-btn nice-btn-secondary">
                                <i class="fas fa-times me-2"></i>İptal
                            </a>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Müşteri seçildiğinde araçlarını yükle
function loadCustomerVehicles() {
    const customerId = document.getElementById('customer_id').value;
    const vehicleSelect = document.getElementById('vehicle_id');
    
    if (!customerId) {
        vehicleSelect.innerHTML = '<option value="">Önce müşteri seçin</option>';
        return;
    }
    
    vehicleSelect.innerHTML = '<option value="">Yükleniyor...</option>';
    
    fetch('get_customer_vehicles.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({ customer_id: customerId })
    })
    .then(response => response.json())
    .then(data => {
        vehicleSelect.innerHTML = '<option value="">Araç seçin</option>';
        
        if (data.success && data.vehicles.length > 0) {
            data.vehicles.forEach(vehicle => {
                const option = document.createElement('option');
                option.value = vehicle.id;
                option.textContent = `${vehicle.brand} ${vehicle.model} (${vehicle.year}) - ${vehicle.plate || 'Plaka Yok'}`;
                vehicleSelect.appendChild(option);
            });
        } else {
            vehicleSelect.innerHTML = '<option value="">Bu müşterinin aracı yok</option>';
        }
    })
    .catch(error => {
        console.error('Error:', error);
        vehicleSelect.innerHTML = '<option value="">Hata oluştu</option>';
    });
}

// Müsait saatleri yükle
function loadAvailableSlots() {
    const date = document.getElementById('appointment_date').value;
    const timeSelect = document.getElementById('appointment_time');
    
    if (!date) {
        timeSelect.innerHTML = '<option value="">Önce tarih seçin</option>';
        return;
    }
    
    timeSelect.innerHTML = '<option value="">Yükleniyor...</option>';
    
    fetch('../get_available_slots.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({ date: date })
    })
    .then(response => response.json())
    .then(data => {
        timeSelect.innerHTML = '<option value="">Saat seçin</option>';
        
        if (data.success && data.slots.length > 0) {
            data.slots.forEach(slot => {
                const option = document.createElement('option');
                option.value = slot;
                option.textContent = slot.substring(0, 5); // HH:MM formatında göster
                timeSelect.appendChild(option);
            });
        } else {
            timeSelect.innerHTML = '<option value="">Bu tarihte müsait saat yok</option>';
        }
    })
    .catch(error => {
        console.error('Error:', error);
        timeSelect.innerHTML = '<option value="">Hata oluştu</option>';
    });
}

// Form validation
(function() {
    'use strict';
    window.addEventListener('load', function() {
        var forms = document.getElementsByClassName('needs-validation');
        var validation = Array.prototype.filter.call(forms, function(form) {
            form.addEventListener('submit', function(event) {
                if (form.checkValidity() === false) {
                    event.preventDefault();
                    event.stopPropagation();
                }
                form.classList.add('was-validated');
            }, false);
        });
    }, false);
})();

// Set minimum date to today
document.getElementById('appointment_date').min = new Date().toISOString().split('T')[0];
</script>

<?php include 'includes/footer.php'; ?>
